## Why is this an issue?

Calling [GetType()](https://learn.microsoft.com/en-us/dotnet/api/system.object.gettype) on a [nullable value type](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/nullable-value-types) object returns
the underlying value type. Therefore, comparing the returned [`Type`](https://learn.microsoft.com/en-us/dotnet/api/system.type)
object to `typeof(Nullable<SomeType>)` will either throw an [NullReferenceException](https://learn.microsoft.com/en-us/dotnet/api/system.nullreferenceexception) or the result will always be
`true` or `false` and can be known at compile time.

## How to fix it

### Code examples

#### Noncompliant code example

    void DoChecks<T>(Nullable<T> value) where T : struct
    {
        bool areEqual = value.GetType() == typeof(Nullable<int>); // Noncompliant: always false
        bool areNotEqual = value.GetType() != typeof(Nullable<int>); // Noncompliant: always true
    
        Nullable<int> nullable = null;
        bool nullComparison = nullable.GetType() != typeof(Nullable<int>); // Noncompliant: throws NullReferenceException
    }

#### Compliant solution

    void DoChecks<T>(Nullable<T> value) where T : struct
    {
        bool areEqual = value.GetType() == typeof(int); // Compliant: can be true or false
        bool areNotEqual = value.GetType() != typeof(int); // Compliant: can be true or false
    
        Nullable<int> nullable = null;
        bool nullComparison = nullable is not null && nullable.GetType() == typeof(int); // Compliant: does not throw NullReferenceException
    }

## Resources

### Documentation

- Microsoft Learn - [Object.GetType Method](https://learn.microsoft.com/en-us/dotnet/api/system.object.gettype)
- Microsoft Learn - [Nullable
  value types (C# reference)](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/nullable-value-types)
- Microsoft Learn - [Type Class](https://learn.microsoft.com/en-us/dotnet/api/system.type)
- Microsoft Learn - [NullReferenceException Class](https://learn.microsoft.com/en-us/dotnet/api/system.nullreferenceexception)