## Why is this an issue?

[Recursion](https://en.wikipedia.org/wiki/Recursion) is a technique used to define a problem in terms of the problem itself, usually in
terms of a simpler version of the problem itself.

For example, the implementation of the generator for the n-th value of the [Fibonacci
sequence](https://en.wikipedia.org/wiki/Fibonacci_sequence) comes naturally from its mathematical definition, when recursion is used:

    int NthFibonacciNumber(int n)
    {
        if (n <= 1)
        {
            return 1;
        }
        else
        {
            return NthFibonacciNumber(n - 1) + NthFibonacciNumber(n - 2);
        }
    }

As opposed to:

    int NthFibonacciNumber(int n)
    {
        int previous = 0;int last = 1;for (var i = 0; i < n; i++){
            (previous, last) = (last, last + previous);}return last;
    }

The use of recursion is acceptable in methods, like the one above, where you can break out of it.

    int NthFibonacciNumber(int n)
    {
        if (n <= 1)
        {
            return 1; // Base case: stop the recursion
        }
        // ...
    }

It is also acceptable and makes sense in some type definitions:

    class Box : IComparable<Box>
    {
        public int CompareTo(Box? other)
        {
            // Compare the two Box instances...
        }
    }

With types, some invalid recursive definitions are caught by the compiler:

    class C2<T> : C2<T>     // Error CS0146: Circular base type dependency
    {
    }
    
    class C2<T> : C2<C2<T>> // Error CS0146: Circular base type dependency
    {
    }

In more complex scenarios, however, the code will compile but execution will result in a [TypeLoadException](https://learn.microsoft.com/en-us/dotnet/api/system.typeloadexception) if you try to instantiate the class.

    class C1<T>
    {
    }
    
    class C2<T> : C1<C2<C2<T>>> // Noncompliant
    {
    }
    
    var c2 = new C2<int>();     // This would result into a TypeLoadException

## Resources

### Documentation

- [Recursion](https://en.wikipedia.org/wiki/Recursion_%28computer_science%29)
- [TypeLoadException](https://learn.microsoft.com/en-us/dotnet/api/system.typeloadexception)

### Articles & blog posts

- [Curiously recurring template pattern](https://en.wikipedia.org/wiki/Curiously_recurring_template_pattern)
- [Modern C# Techniques, Part
  1: Curiously Recurring Generic Pattern](https://blog.stephencleary.com/2022/09/modern-csharp-techniques-1-curiously-recurring-generic-pattern.html)
- [Curiouser and curiouser](https://ericlippert.com/2011/02/02/curiouser-and-curiouser/)