## Why is this an issue?

When a class has only a `private` constructor, it can’t be instantiated except within the class itself. Such classes can be considered
[dead code](https://en.wikipedia.org/wiki/Dead_code) and should be fixed

### Exceptions

- Classes that access their private constructors ([singletons](https://en.wikipedia.org/wiki/Singleton_pattern) or [smart
  enums](https://learn.microsoft.com/en-us/dotnet/architecture/microservices/microservice-ddd-cqrs-patterns/enumeration-classes-over-enum-types)) are ignored.
- Classes with only `static` members are also ignored because they are covered by Rule {rule:csharpsquid:S1118}.
- Classes that derive from [SafeHandle](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.safehandle) since
  they can be instantiate through [P/Invoke](https://learn.microsoft.com/en-us/dotnet/standard/native-interop/pinvoke).

## How to fix it

### Code examples

#### Noncompliant code example

    public class MyClass // Noncompliant: the class contains only private constructors
    {
      private MyClass() { ... }
    }

#### Compliant solution

    public class MyClass // Compliant: the class contains at least one non-private constructor
    {
      public MyClass() { ... }
    }

## Resources

### Documentation

- {rule:csharpsquid:S1118} - Utility classes should not have public constructors
- Microsoft Learn - [SafeHandle Class](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.safehandle)
- Microsoft Learn - [Platform Invoke (P/Invoke)](https://learn.microsoft.com/en-us/dotnet/standard/native-interop/pinvoke)
- Microsoft Learn - [Use
  enumeration classes instead of enum types](https://learn.microsoft.com/en-us/dotnet/architecture/microservices/microservice-ddd-cqrs-patterns/enumeration-classes-over-enum-types)
- Wikipedia - [Dead code](https://en.wikipedia.org/wiki/Dead_code)
- Wikipedia - [Singleton pattern](https://en.wikipedia.org/wiki/Singleton_pattern)

### Articles & blog posts

- C# in Depth - [Implementing the Singleton Pattern in C#](https://csharpindepth.com/articles/singleton)
- Medium - [Making enums smarter in C#](https://medium.com/null-exception/making-enums-smarter-in-c-518108cdaa73)