## Why is this an issue?

Calling `GetType` on a `Type` variable will always return the `System.Type` representation, which is equivalent to
`typeof(System.Type)`. This also applies to passing a `Type` argument to `IsInstanceOfType` which always returns
`false`.

In both cases, the results are entirely predictable and should be avoided.

### Exceptions

Calling `GetType` on `System.Type` is considered compliant to get an instance of `System.RuntimeType`, as
demonstrated in the following example:

    typeof(Type).GetType(); // Can be used by convention to get an instance of 'System.RuntimeType'

## How to fix it

Make sure the usage of `GetType` or `IsInstanceOfType` is invoked with the correct variable or argument type.

### Code examples

#### Noncompliant code example

    void ExamineSystemType(string str)
    {
        Type stringType = str.GetType();
        Type runtimeType = stringType.GetType(); // Noncompliant
    
        if (stringType.IsInstanceOfType(typeof(string))) // Noncompliant; will always return false
        { /* ... */ }
    }

#### Compliant solution

    void ExamineSystemType(string str)
    {
        Type stringType = str.GetType();
    
        if (stringType.IsInstanceOfType(str)) // Compliant
        { /* ... */ }
    }

## Resources

### Documentation

- ["Type" class](https://learn.microsoft.com/en-us/dotnet/api/system.type)
- ["GetType" Method](https://learn.microsoft.com/en-us/dotnet/api/system.object.gettype)
- ["IsInstanceOfType" Method](https://learn.microsoft.com/en-us/dotnet/api/system.type.isinstanceoftype)

### Articles & blog posts

- [Difference between "System.Type" and "System.RuntimeType"](https://stackoverflow.com/a/5737947)