## Why is this an issue?

It is a well-established convention to name each logger after its enclosing type. This rule raises an issue when the convention is not
respected.

    class EnclosingType
    {
        private readonly ILogger logger;
    
        public EnclosingType(ILoggerFactory loggerFactory)
        {
            logger = loggerFactory.CreateLogger<AnotherType>();   // Noncompliant
            logger = loggerFactory.CreateLogger<EnclosingType>(); // Compliant
        }
    }

Not following such a convention can result in confusion and logging misconfiguration.

For example, the person configuring the log may attempt to change the logging behavior for the `MyNamespace.EnclosingType` type, by
overriding defaults for the logger named after that type.

    {
        "Logging": {
            "LogLevel": {
                "Default": "Error",
                "MyNamespace.EnclosingType": "Debug"
            }
        }
    }

However, if the convention is not in place, the override would not affect logs from `MyNamespace.EnclosingType`, since they are made via
a logger with a different name.

Moreover, using the same logger name for multiple types prevents the granular configuration of each type’s logger, since there is no way to
distinguish them in configuration.

The rule targets the following logging frameworks: \* [Microsoft
Extensions Logging](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging) \* [Apache log4net](https://logging.apache.org/log4net/) \* [NLog](https://nlog-project.org/)

### Exceptions

The rule doesn’t raise issues when custom handling of logging names is in place, and the logger name is not derived from a `Type`.

    class EnclosingType
    {
        private readonly ILogger logger;
    
        EnclosingType(ILoggerFactory loggerFactory)
        {
            logger = loggerFactory.CreateLogger("My cross-type logging category");   // Compliant
            logger = loggerFactory.CreateLogger(AComplexLogicToFindTheRightType());  // Compliant
        }
    }

## How to fix it

When the logger name is defined by a generic type parameter:

    class EnclosingType
    {
        private readonly ILogger logger;
    
        public EnclosingType(ILoggerFactory loggerFactory)
        {
            logger = loggerFactory.CreateLogger<AnotherType>();   // Noncompliant
            logger = loggerFactory.CreateLogger<EnclosingType>(); // Compliant
        }
    }

When the logger name is defined by an input parameter of type `Type`:

    class EnclosingType
    {
        private readonly ILogger logger;
    
        public EnclosingType(ILoggerFactory loggerFactory)
        {
            logger = loggerFactory.CreateLogger(typeof(AnotherType));   // Noncompliant
            logger = loggerFactory.CreateLogger(typeof(EnclosingType)); // Compliant
            logger = loggerFactory.CreateLogger(GetType());             // Compliant
        }
    }

When the logger name is a string, derived from a `Type`:

    class EnclosingType
    {
        private readonly ILogger logger;
    
        public EnclosingType(ILoggerFactory loggerFactory)
        {
            logger = loggerFactory.CreateLogger(typeof(AnotherType).Name);       // Noncompliant
            logger = loggerFactory.CreateLogger(typeof(AnotherType).FullName);   // Noncompliant
            logger = loggerFactory.CreateLogger(nameof(AnotherType));            // Noncompliant
            // Fix by referring to the right type
            logger = loggerFactory.CreateLogger(typeof(EnclosingType).Name);     // Compliant
            logger = loggerFactory.CreateLogger(typeof(EnclosingType).FullName); // Compliant
            logger = loggerFactory.CreateLogger(nameof(EnclosingType));          // Compliant
            // or by retrieving the right type dynamically
            logger = loggerFactory.CreateLogger(GetType().FullName);             // Compliant
        }
    }

## Resources

### Documentation

- Microsoft Learn - [.NET logging and tracing](https://learn.microsoft.com/en-us/dotnet/core/diagnostics/logging-tracing)
- Microsoft Learn - [Logging in C# and
  .NET - Log category](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging?tabs=command-line#log-category)
- Microsoft Learn - [Logging in C#
  and .NET - Configure logging](https://learn.microsoft.com/en-us/dotnet/core/extensions/logging?tabs=command-line#configure-logging)
- Microsoft Learn - [ILogger&lt;TCategoryName&gt;
  Interface](https://learn.microsoft.com/en-us/dotnet/api/microsoft.extensions.logging.ilogger-1)
- Apache Logging - [Apache log4net](https://logging.apache.org/log4net/)
- NLog - [Flexible & free open-source logging for .NET](https://nlog-project.org/)

### Articles & blog posts

- Raygun Blog - [C# logging: Best practices in 2023 with examples and tools](https://raygun.com/blog/c-sharp-logging-best-practices/)
- Apache Logging - [Apache log4net Manual - Configuration](https://logging.apache.org/log4net/release/manual/configuration.html)
- GitHub NLog repository - [Best practices for using NLog](https://github.com/nlog/nlog/wiki/Tutorial#best-practices-for-using-nlog)