## Why is this an issue?

An assertion is a piece of code that’s used during development when the [compilation debug mode is activated](https://learn.microsoft.com/en-us/visualstudio/debugger/how-to-set-debug-and-release-configurations). It
allows a program to check itself as it runs. When an assertion is `true`, that means everything is operating as expected.

In non-debug mode, all [`Debug.Assert`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.debug.assert) calls
are automatically left out (via the [`Conditional("DEBUG")`](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.conditionalattribute) mechanism). So, by
contract, the boolean expressions that are evaluated by those assertions must not contain any [side effects](https://en.wikipedia.org/wiki/Side_effect_%28computer_science%29). Otherwise, when leaving the debug mode, the functional behavior
of the application is not the same anymore.

The rule will raise if the method name starts with any of the following `remove`, `delete`, `add`,
`pop`, `update`, `retain`, `insert`, `push`, `append`, `clear`,
`dequeue`, `enqueue`, `dispose`, `put`, or `set`, although `SetEquals` will be
ignored.

## How to fix it

In the following example, the assertion checks the return value of the remove method in the argument. Because the whole line is skipped in
non-debug builds, the call to `Remove` never happens in such builds.

### Code examples

#### Noncompliant code example

    Debug.Assert(list.Remove("dog"));

#### Compliant solution

The `Remove` call must be extracted and the return value needs to be asserted instead.

    bool result = list.Remove("dog");
    Debug.Assert(result);

## Resources

### Documentation

- Microsoft Learn [`Debug.Assert` Method](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.debug.assert/)
- Microsoft Learn [Debugging, tracing, and profiling](https://learn.microsoft.com/en-us/dotnet/framework/debug-trace-profile/)
- Microsoft Learn [How to: Compile
  Conditionally with Trace and Debug](https://learn.microsoft.com/en-us/dotnet/framework/debug-trace-profile/how-to-compile-conditionally-with-trace-and-debug)
- Microsoft Learn [Miscellaneous attributes
  interpreted by the C# compiler - `Conditional` attribute](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/attributes/general#conditional-attribute)

### Articles & blog posts

- Wikipedia [Side effect (computer science)](https://en.wikipedia.org/wiki/Side_effect_%28computer_science%29)