When a cookie is configured with the `HttpOnly` attribute set to *true*, the browser guaranties that no client-side script will
be able to read it. In most cases, when a cookie is created, the default value of `HttpOnly` is *false* and it’s up to the developer
to decide whether or not the content of the cookie can be read by the client-side script. As a majority of Cross-Site Scripting (XSS) attacks target
the theft of session-cookies, the `HttpOnly` attribute can help to reduce their impact as it won’t be possible to exploit the XSS
vulnerability to steal session-cookies.

## Ask Yourself Whether

- the cookie is sensitive, used to authenticate the user, for instance a *session-cookie*
- the `HttpOnly` attribute offer an additional protection (not the case for an *XSRF-TOKEN cookie* / CSRF token for example)

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- By default the `HttpOnly` flag should be set to *true* for most of the cookies and it’s mandatory for session /
  sensitive-security cookies.

## Sensitive Code Example

When the `HttpCookie.HttpOnly` property is set to `false` then the cookie can be accessed by client side code:

    HttpCookie myCookie = new HttpCookie("Sensitive cookie");
    myCookie.HttpOnly = false; // Sensitive: this cookie is created with the httponly flag set to false and so it can be stolen easily in case of XSS vulnerability

The [default value](https://docs.microsoft.com/en-us/dotnet/api/system.web.httpcookie.httponly?view=netframework-4.8) of
`HttpOnly` flag is `false`, unless overwritten by an application’s configuration file:

    HttpCookie myCookie = new HttpCookie("Sensitive cookie");
    // Sensitive: this cookie is created without the httponly flag  (by default set to false) and so it can be stolen easily in case of XSS vulnerability

## Compliant Solution

Set the `HttpCookie.HttpOnly` property to `true`:

    HttpCookie myCookie = new HttpCookie("Sensitive cookie");
    myCookie.HttpOnly = true; // Compliant: the sensitive cookie is protected against theft thanks to the HttpOnly property set to true (HttpOnly = true)

Or change the default flag values for the whole application by editing the [Web.config configuration file](https://docs.microsoft.com/en-us/previous-versions/dotnet/netframework-4.0/ms228262%28v=vs.100%29):

    <httpCookies httpOnlyCookies="true" requireSSL="true" />

- the `requireSSL` attribute corresponds programmatically to the `Secure` field.
- the `httpOnlyCookies` attribute corresponds programmatically to the `httpOnly` field.

## See

- OWASP - [Top 10 2021 Category A5 - Security Misconfiguration](https://owasp.org/Top10/A05_2021-Security_Misconfiguration/)
- [OWASP HttpOnly](https://owasp.org/www-community/HttpOnly)
- OWASP - [Top 10 2017 Category A7 - Cross-Site Scripting
  (XSS)](https://owasp.org/www-project-top-ten/2017/A7_2017-Cross-Site_Scripting_%28XSS%29)
- CWE - [CWE-1004 - Sensitive Cookie Without 'HttpOnly' Flag](https://cwe.mitre.org/data/definitions/1004)
- Derived from FindSecBugs rule [HTTPONLY_COOKIE](https://find-sec-bugs.github.io/bugs.htm#HTTPONLY_COOKIE)
- STIG Viewer - [Application Security and
  Development: V-222575](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222575) - The application must set the HTTPOnly flag on session cookies.