This vulnerability exposes encrypted data to a number of attacks whose goal is to recover the plaintext.

## Why is this an issue?

Encryption algorithms are essential for protecting sensitive information and ensuring secure communications in a variety of domains. They are used
for several important reasons:

- Confidentiality, privacy, and intellectual property protection
- Security during transmission or on storage devices
- Data integrity, general trust, and authentication

When selecting encryption algorithms, tools, or combinations, you should also consider two things:

1. No encryption is unbreakable.
2. The strength of an encryption algorithm is usually measured by the effort required to crack it within a reasonable time frame.

In the mode Cipher Block Chaining (CBC), each block is used as cryptographic input for the next block. For this reason, the first block requires an
initialization vector (IV), also called a "starting variable" (SV).

If the same IV is used for multiple encryption sessions or messages, each new encryption of the same plaintext input would always produce the same
ciphertext output. This may allow an attacker to detect patterns in the ciphertext.

### What is the potential impact?

After retrieving encrypted data and performing cryptographic attacks on it on a given timeframe, attackers can recover the plaintext that
encryption was supposed to protect.

Depending on the recovered data, the impact may vary.

Below are some real-world scenarios that illustrate the potential impact of an attacker exploiting the vulnerability.

#### Additional attack surface

By modifying the plaintext of the encrypted message, an attacker may be able to trigger additional vulnerabilities in the code. An attacker can
further exploit a system to obtain more information.  
 Encrypted values are often considered trustworthy because it would not be possible for a
third party to modify them under normal circumstances.

#### Breach of confidentiality and privacy

When encrypted data contains personal or sensitive information, its retrieval by an attacker can lead to privacy violations, identity theft,
financial loss, reputational damage, or unauthorized access to confidential systems.

In this scenario, a company, its employees, users, and partners could be seriously affected.

The impact is twofold, as data breaches and exposure of encrypted data can undermine trust in the organization, as customers, clients and
stakeholders may lose confidence in the organization’s ability to protect their sensitive data.

#### Legal and compliance issues

In many industries and locations, there are legal and compliance requirements to protect sensitive data. If encrypted data is compromised and the
plaintext can be recovered, companies face legal consequences, penalties, or violations of privacy laws.

## How to fix it in .NET

### Code examples

#### Noncompliant code example

    using System.IO;
    using System.Security.Cryptography;
    
    public void Encrypt(byte[] key, byte[] dataToEncrypt, MemoryStream target)
    {
        var aes = new AesCryptoServiceProvider();
    
        byte[] iv     = new byte[] { 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16 };
        var encryptor = aes.CreateEncryptor(key, iv); // Noncompliant
    
        var cryptoStream = new CryptoStream(target, encryptor, CryptoStreamMode.Write);
        var swEncrypt    = new StreamWriter(cryptoStream);
    
        swEncrypt.Write(dataToEncrypt);
    }

#### Compliant solution

In this example, the code implicitly uses a number generator that is considered **strong**, thanks to `aes.IV`.

    using System.IO;
    using System.Security.Cryptography;
    
    public void Encrypt(byte[] key, byte[] dataToEncrypt, MemoryStream target)
    {
        var aes = new AesCryptoServiceProvider();
    
        var encryptor = aes.CreateEncryptor(key, aes.IV);
    
        var cryptoStream = new CryptoStream(target, encryptor, CryptoStreamMode.Write);
        var swEncrypt    = new StreamWriter(cryptoStream);
    
        swEncrypt.Write(dataToEncrypt);
    }

### How does this work?

#### Use unique IVs

To ensure high security, initialization vectors must meet two important criteria:

- IVs must be unique for each encryption operation.
- For CBC and CFB modes, a secure FIPS-compliant random number generator should be used to generate unpredictable IVs.

The IV does not need be secret, so the IV or information sufficient to determine the IV may be transmitted along with the ciphertext.

In the previous non-compliant example, the problem is not that the IV is hard-coded.  
 It is that the same IV is used for multiple encryption
attempts.

## Resources

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- OWASP - [Top 10 2017 Category A6 - Security
  Misconfiguration](https://owasp.org/www-project-top-ten/2017/A6_2017-Security_Misconfiguration)
- CWE - [CWE-329 - Not Using an Unpredictable IV with CBC Mode](https://cwe.mitre.org/data/definitions/329)
- CWE - [CWE-780 - Use of RSA Algorithm without OAEP](https://cwe.mitre.org/data/definitions/780)
- [NIST, SP-800-38A](https://nvlpubs.nist.gov/nistpubs/Legacy/SP/nistspecialpublication800-38a.pdf) - Recommendation for Block Cipher
  Modes of Operation