## Why is this an issue?

[Enumerations](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/enum) are commonly used to identify
distinct elements from a set of values.

However, they can also serve as [bit flags](https://en.wikipedia.org/wiki/Bit_field), enabling bitwise operations to combine multiple
elements within a single value.

    // Saturday = 0b00100000, Sunday = 0b01000000, weekend = 0b01100000
    var weekend = Days.Saturday | Days.Sunday;  // Combining elements

When enumerations are used as bit flags, it is considered [good practice](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/enum#enumeration-types-as-bit-flags) to
annotate the enum type with the [FlagsAttribute](https://learn.microsoft.com/en-us/dotnet/api/system.flagsattribute):

    enum Permissions
    {
      None = 0,
      Read = 1,
      Write = 2,
      Execute = 4
    }
    
    // ...
    
    var x = Permissions.Read | Permissions.Write;  // Noncompliant: enum is not annotated with [Flags]

The `FlagsAttribute` explicitly marks an enumeration as bit flags, making it clear that it uses bit fields and is intended to be used as
flags.

    [Flags]
    enum Permissions
    {
      None = 0,
      Read = 1,
      Write = 2,
      Execute = 4
    }
    
    // ...
    
    var x = Permissions.Read | Permissions.Write;  // Compliant: enum is annotated with [Flags]

Additionally, adding the `FlagsAttribute` to the enumeration enable a [better string representation](https://learn.microsoft.com/en-us/dotnet/api/system.flagsattribute#examples) when using the [Enum.ToString](https://learn.microsoft.com/en-us/dotnet/api/system.enum.tostring) method.

## Resources

### Documentation

- [Enumeration in C#](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/enum)
    - [Enumeration
      types as bit flags](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/enum#enumeration-types-as-bit-flags)
- [FlagsAttribute class](https://learn.microsoft.com/en-us/dotnet/api/system.flagsattribute)
- [Enum.ToString method](https://learn.microsoft.com/en-us/dotnet/api/system.enum.tostring)
- [Bit field - Wikipedia](https://en.wikipedia.org/wiki/Bit_field)