## Why is this an issue?

The [foreach](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/iteration-statements#the-foreach-statement)
statement was introduced in the C# language prior to generics to make it easier to work with the non-generic collections available at that time such
as [ArrayList](https://learn.microsoft.com/en-us/dotnet/api/system.collections.arraylist). The `foreach` statements allow you to
downcast elements of a collection of [Objects](https://learn.microsoft.com/en-us/dotnet/api/system.object) to any other type.

The problem is that to achieve the cast, the `foreach` statements silently perform [explicit type
conversion](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/types/casting-and-type-conversions#explicit-conversions), which at runtime can result in an [InvalidCastException](https://learn.microsoft.com/en-us/dotnet/api/system.invalidcastexception).

C# code iterating on generic collections or arrays should not rely on `foreach` statement’s silent `explicit`
conversions.

    public class Fruit { }
    public class Orange : Fruit { }
    public class Apple : Fruit { }
    
    class MyTest
    {
      public void Test()
      {
        var fruitBasket = new List<Fruit>();
        fruitBasket.Add(new Orange());
        fruitBasket.Add(new Orange());
        fruitBasket.Add(new Apple());
    
        foreach (Orange orange in fruitBasket) // Noncompliant
        {
          //...
        }
      }
    }

    public class Fruit { }
    public class Orange : Fruit { }
    public class Apple : Fruit { }
    
    class MyTest
    {
      public void Test()
      {
        var fruitBasket = new List<Fruit>();
        fruitBasket.Add(new Orange());
        fruitBasket.Add(new Orange());
        fruitBasket.Add(new Apple());
    
        foreach (Orange orange in fruitBasket.OfType<Orange>())
        {
          //...
        }
      }
    }

### Exceptions

The rule ignores iterations on collections of `objects`. This includes legacy code that uses `ArrayList`. Furthermore, the
rule does not report on cases when user-defined conversions are being called.

## Resources

### Documentation

- [Foreach
  statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/iteration-statements#the-foreach-statement)
- [ArrayList](https://learn.microsoft.com/en-us/dotnet/api/system.collections.arraylist)
- [Object class](https://learn.microsoft.com/en-us/dotnet/api/system.object)
- [Explicit
  conversion](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/types/casting-and-type-conversions#explicit-conversions)
- [InvalidCastException](https://learn.microsoft.com/en-us/dotnet/api/system.invalidcastexception)