## Why is this an issue?

An [`async`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/async) method with a
`void` return type does not follow the [task asynchronous programming
(TAP)](https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/task-asynchronous-programming-model) model since the return type should be [`Task`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task) or [`Task<TResult>`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task-1)

Doing so prevents control over the [asynchronous
execution](https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/async-scenarios), such as:

- waiting for the execution to complete
- catching any exception that might occur during execution
- testing execution behavior

### Exceptions

- Methods with the [`EventHandler`](https://learn.microsoft.com/en-us/dotnet/api/system.eventhandler) delegate signature.  Using `void` for `EventHandler` is compliant with the TAP model.  

        public async void button1_Click(object sender, EventArgs e)
        {
          await DoSomethingAsync();
        }
- Methods name matching `On[A-Z]\w*` pattern.  Some frameworks may not use the same `EventHandler` method signature  

        public async void OnClick(EventContext data)
        {
          await DoSomethingAsync();
        }

## How to fix it

Update the return type of the method from `void` to `Task`.

### Code examples

#### Noncompliant code example

    private async void ThrowExceptionAsync() // Noncompliant: async method return type is 'void'
    {
      throw new InvalidOperationException();
    }
    
    public void Method()
    {
      try
      {
        ThrowExceptionAsync();
      }
      catch (Exception)
      {
        // The exception is never caught here
        throw;
      }
    }

#### Compliant solution

    private async Task ThrowExceptionAsync() // Compliant: async method return type is 'Task'
    {
      throw new InvalidOperationException();
    }
    
    public async Task Method()
    {
      try
      {
        await ThrowExceptionAsync();
      }
      catch (Exception)
      {
        // The exception is caught here
        throw;
      }
    }

## Resources

### Documentation

- Microsoft Learn - [`async` (C#
  Reference)](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/async)
- Microsoft Learn - [Asynchronous
  programming](https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/async-scenarios)
- Microsoft Learn - [Task
  asynchronous programming model](https://learn.microsoft.com/en-us/dotnet/csharp/asynchronous-programming/task-asynchronous-programming-model)
- Microsoft Learn - [`Task` Class](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task)
- Microsoft Learn - [`Task<TResult>`
  Class](https://learn.microsoft.com/en-us/dotnet/api/system.threading.tasks.task-1)
- Microsoft Learn - [`EventHandler` Delegate](https://learn.microsoft.com/en-us/dotnet/api/system.eventhandler)