## Why is this an issue?

One of the possible ways of performing [type-testing](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast) is via the [is operator](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/is): `food is Pizza`.

The `is` operator is often used before a direct [cast](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#cast-expression) to the target type,
as a more flexible and powerful alternative to the [as operator](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#as-operator), especially when
used to perform [pattern
matching](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast#type-testing-with-pattern-matching).

    if (food is Pizza pizza)

There’s no valid reason to test `this` with `is`. The only plausible explanation for such a test is that you’re executing
code in a parent class conditionally based on the kind of child class `this` is.

    public class Food
    {
      public void DoSomething()
      {
        if (this is Pizza) // Noncompliant
        {
          // Code specific to Pizza...
        }
      }
    }

However, code that’s specific to a child class should be *in* that child class, not in the parent.

## How to fix it

One way is to take advantage of the [object-orientation](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/tutorials/oop) of
C# and use [polymorphism](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/object-oriented/polymorphism).

- Make the method [virtual](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/virtual), if it is not
  already. That will allow derived classes to perform [method overriding](https://en.wikipedia.org/wiki/Method_overriding).
- Move the code to the right level of the type hierarchy.
- Use [base](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/base) to call the method on the base class
  that has been overridden.

For example, when simple method polymorphism is not enough because it is necessary to reuse multiple sections of the parent method, the [Template method pattern](https://en.wikipedia.org/wiki/Template_method_pattern) might help.

### Code examples

#### Noncompliant code example

    public class Food
    {
      public void DoSomething()
      {
        // Code shared by all Food...
        if (this is Pizza) // Noncompliant
        {
          // Code specific to Pizza...
        }
      }
    }

#### Compliant solution

    public class Food
    {
      public virtual void DoSomething()
      {
        // Code shared by all Food...
      }
    }
    
    public class Pizza : Food
    {
      public override void DoSomething()
      {
        base.DoSomething();
        // Code specific to Pizza...
      }
    }

## Resources

### Documentation

- [Type-testing operators and cast
  expressions - is, as, typeof and casts](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/type-testing-and-cast)
- [Pattern matching overview](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/functional/pattern-matching)
- [Object-Oriented programming (C#)](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/tutorials/oop)
- [Polymorphism](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/object-oriented/polymorphism)
- [Template method pattern](https://en.wikipedia.org/wiki/Template_method_pattern)
- [Method overriding](https://en.wikipedia.org/wiki/Method_overriding)