## Why is this an issue?

In C#, the [`Object.ReferenceEquals`](https://learn.microsoft.com/en-us/dotnet/api/system.object.referenceequals) method is
used to compare two [reference type](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/reference-types)
variables. If you use this method to compare two [value types](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/value-types), such as `int`,
`float`, or `bool` you will not get the expected results because value type variables contain an instance of the type and not a
reference to it.

Due to value type variables containing directly an instance of the type, they can’t have the same reference, and using
`Object.ReferenceEquals` to compare them will always return `false` even if the compared variables have the same value.

## How to fix it

When comparing value types, prefer using the [`Object.Equals`](https://learn.microsoft.com/en-us/dotnet/api/system.object.equals).

Note that in the case of [structure types](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/struct), it
is recommended to [implement
value equality](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/statements-expressions-operators/how-to-define-value-equality-for-a-type#struct-example). If not, {rule:csharpsquid:S3898} might raise.

### Code examples

#### Noncompliant code example

    using System;
    
    struct MyStruct
    {
        int valueA;
        int valueB;
    }
    
    static class MyClass
    {
        public static void Method(MyStruct struct1, MyStruct struct2)
        {
            if (Object.ReferenceEquals(struct1, struct2)) // Noncompliant: this will be always false
            {
                // ...
            }
        }
    }

#### Compliant solution

    using System;
    
    struct MyStruct : IEquatable<MyStruct>
    {
        int valueA;
        int valueB;
    
        public bool Equals(MyStruct other) => valueA == other.valueA && valueB == other.valueB;
    
        public override bool Equals(object obj) => obj is MyStruct other && Equals(other);
    
        public override int GetHashCode() => HashCode.Combine(valueA, valueB);
    
        public static bool operator ==(MyStruct lhs, MyStruct rhs) => lhs.Equals(rhs);
    
        public static bool operator !=(MyStruct lhs, MyStruct rhs) => !(lhs == rhs);
    }
    
    static class MyClass
    {
        public static void Method(MyStruct struct1, MyStruct struct2)
        {
            if (struct1.Equals(struct2)) // Compliant: value are compared
            {
                // ...
            }
        }
    }

## Resources

### Documentation

- Microsoft Learn - [`Object.ReferenceEquals(Object,
  Object)` Method](https://learn.microsoft.com/en-us/dotnet/api/system.object.referenceequals)
- Microsoft Learn - [`Object.Equals` Method](https://learn.microsoft.com/en-us/dotnet/api/system.object.equals)
- Microsoft Learn - [Value types (C#
  reference)](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/value-types)
- Microsoft Learn - [Reference types (C#
  reference)](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/reference-types)
- Microsoft Learn - [Equality operators
  - test if two objects are equal or not](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/equality-operators)
- Microsoft Learn - [How to define value equality for a class or struct (C# Programming Guide)](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/statements-expressions-operators/how-to-define-value-equality-for-a-type#struct-example)
- Microsoft Learn - [Structure types (C#
  reference)](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/struct)
- {rule:csharpsquid:S3898} - Value types should implement "IEquatable&lt;T&gt;"