## Why is this an issue?

In C#, without constraints on a generic type parameter, both [reference](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/reference-types) and [value](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/value-types) types can be passed. However, comparing
this type parameter to `null` can be misleading as value types, like `struct`, can never be null.

## How to fix it

To avoid unexpected comparisons:

- if you expect a value type, use [default()](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/default#default-operator) for comparison
- if you expect a reference type, add a [constraint](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/generics/constraints-on-type-parameters) to prevent value
  types from being passed

### Code examples

#### Noncompliant code example

    bool IsDefault<T>(T value)
    {
      if (value == null) // Noncompliant
      {
        // ...
      }
    }

#### Compliant solution

    bool IsDefault<T>(T value)
    {
      if (EqualityComparer<T>.Default.Equals(value, default(T)))
      {
        // ...
      }
    }

or

    bool IsDefault<T>(T value) where T : class
    {
      if (value == null)
      {
        // ...
      }
    }

## Resources

### Documentation

- Microsoft Learn - [Constraints on type parameters](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/generics/constraints-on-type-parameters)
- Microsoft Learn - [Reference types](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/reference-types)
- Microsoft Learn - [Value types](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/value-types)
- Microsoft Learn - [`default` operator](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/default#default-operator)