## Why is this an issue?

An `IDisposable` object should be disposed (there are some rare exceptions where not disposing is fine, most notably `Task`).
If a class has an `IDisposable` field, there can be two situations:

- The class observes a field that is under the responsibility of another class.
- The class owns the field, and is therefore responsible for calling `Dispose` on it.

In the second case, the safest way for the class to ensure `Dispose` is called is to call it in its own `Dispose` function,
and therefore to be itself `IDisposable`. A class is considered to own an `IDisposable` field resource if it created the object
referenced by the field.

### Noncompliant code example

    public class ResourceHolder   // Noncompliant; doesn't implement IDisposable
    {
        private FileStream fs;  // This member is never Disposed
        public void OpenResource(string path)
        {
            this.fs = new FileStream(path, FileMode.Open); // I create the FileStream, I'm owning it
        }
        public void CloseResource()
        {
            this.fs.Close();
        }
    }

### Compliant solution

    public class ResourceHolder : IDisposable
    {
        private FileStream fs;
        public void OpenResource(string path)
        {
            this.fs = new FileStream(path, FileMode.Open); // I create the FileStream, I'm owning it
        }
        public void CloseResource()
        {
            this.fs.Close();
        }
        public void Dispose()
        {
            this.fs.Dispose();
        }
    }

## Resources

- CWE - [CWE-459 - Incomplete Cleanup](https://cwe.mitre.org/data/definitions/459)