## Why is this an issue?

When writing [managed code](https://learn.microsoft.com/en-us/dotnet/standard/managed-code), there is no need to worry about memory
allocation or deallocation as it is taken care of by the [garbage
collector](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection). However, certain objects, such as `Bitmap`, utilize unmanaged memory for specific purposes like [pointer arithmetic](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/unsafe-code). These objects may have substantial
unmanaged memory footprints while having minimal managed footprints. Unfortunately, the garbage collector only recognizes the small managed footprint
and does not promptly reclaim the corresponding unmanaged memory (by invoking the finalizer method of `Bitmap`) for efficiency reasons.

In addition, it’s essential to manage other system resources besides memory. The operating system has limits on the number of [file descriptors](https://en.wikipedia.org/wiki/File_descriptor) (e.g., `FileStream`) or [sockets](https://en.wikipedia.org/wiki/Network_socket) (e.g., `WebClient`) that can remain open simultaneously. Therefore, it’s
crucial to `Dispose` of these resources promptly when they are no longer required, instead of relying on the garbage collector to invoke
the finalizers of these objects at an unpredictable time in the future.

This rule keeps track of `private` fields and local variables of specific types that implement `IDisposable` or
`IAsyncDisposable`. It identifies instances of these types that are not properly disposed, closed, aliased, returned, or passed to other
methods. This applies to instances that are either directly created using the `new` operator or instantiated through a predefined list of
factory methods.

Here is the list of predefined factory methods tracked by this rule:

- `System.IO.File.Create()`
- `System.IO.File.Open()`
- `System.Drawing.Image.FromFile()`
- `System.Drawing.Image.FromStream()`

### Exceptions

`IDisposable` / `IAsyncDisposable` variables returned from a method or passed to other methods are ignored, as are local
`IDisposable` / `IAsyncDisposable` objects that are initialized with other `IDisposable` /
`IAsyncDisposable` objects.

    public Stream WriteToFile(string path, string text)
    {
      var fs = new FileStream(path, FileMode.Open); // Compliant: it is returned
      var bytes = Encoding.UTF8.GetBytes(text);
      fs.Write(bytes, 0, bytes.Length);
      return fs;
    }
    
    public void ReadFromStream(Stream s)
    {
      var sr = new StreamReader(s); // Compliant: it would close the underlying stream.
      // ...
    }

## How to fix it

It is essential to identify what kind of disposable resource variable is used to know how to fix this issue.

In the case of a disposable resource store as a member (either as field or property), it should be disposed at the same time as the class. The best
way to achieve this is to follow the [dispose
pattern](https://learn.microsoft.com/en-us/dotnet/standard/design-guidelines/dispose-pattern).

When creating the disposable resource for a one-time use (cases not covered by the exceptions), it should be disposed at the end of its creation
scope. The easiest to ensure your resource is disposed when reaching the end of a scope is to either use [the using statement or the using declaration](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/using)

### Code examples

#### Noncompliant code example

    public class ResourceHolder
    {
      private FileStream fs; // Noncompliant: dispose or close are never called
    
      public void OpenResource(string path)
      {
        this.fs = new FileStream(path, FileMode.Open);
      }
    
      public void WriteToFile(string path, string text)
      {
        var fs = new FileStream(path, FileMode.Open); // Noncompliant: not disposed, returned or initialized with another disposable object
        var bytes = Encoding.UTF8.GetBytes(text);
        fs.Write(bytes, 0, bytes.Length);
      }
    }

#### Compliant solution

    public class ResourceHolder : IDisposable, IAsyncDisposable
    {
      private FileStream fs; // Compliant: disposed in Dispose/DisposeAsync methods
    
      public void OpenResource(string path)
      {
        this.fs = new FileStream(path, FileMode.Open);
      }
    
      public void Dispose()
      {
        this.fs.Dispose();
      }
    
      public async ValueTask DisposeAsync()
      {
        await fs.DisposeAsync().ConfigureAwait(false);
      }
    
      public void WriteToFile(string path, string text)
      {
        using (var fs = new FileStream(path, FileMode.Open)) // Compliant: disposed at the end of the using block
        {
          var bytes = Encoding.UTF8.GetBytes(text);
          fs.Write(bytes, 0, bytes.Length);
        }
      }
    }

## Resources

### Documentation

- [What is "managed code"?](https://learn.microsoft.com/en-us/dotnet/standard/managed-code)
- [Garbage collection](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection)
- [Finalizers](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/finalizers)
- [Unsafe code, pointer types, and function pointers](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/unsafe-code)
- [File descriptor - Wiki](https://en.wikipedia.org/wiki/File_descriptor)
- [Network socket - Wiki](https://en.wikipedia.org/wiki/Network_socket)
- [Dispose pattern](https://learn.microsoft.com/en-us/dotnet/standard/design-guidelines/dispose-pattern)
    - [Implement a Dispose method](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/implementing-dispose)
    - [Implement a DisposeAsync
      method](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/implementing-disposeasync)
- [using statement and using declaration](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/using)
- CWE - [CWE-459 - Incomplete Cleanup](https://cwe.mitre.org/data/definitions/459)