## Why is this an issue?

Most checks against an [IndexOf](https://learn.microsoft.com/en-us/dotnet/api/system.string.indexof) value compare it with -1 because
**0 is a valid index**.

    strings.IndexOf(someString) == -1 // Test for "index not found"
    strings.IndexOf(someString) < 0   // Test for "index not found"
    strings.IndexOf(someString) >= 0  // Test for "index found"

Any checks which look for values `> 0` ignore the first element, which is likely a bug. If the intent is merely to check the
inclusion of a value in a `string`, `List`, or array, consider using the [Contains](https://learn.microsoft.com/en-us/dotnet/api/system.string.contains) method instead.

    strings.Contains(someString) // bool result

This rule raises an issue when the output value of any of the following methods is tested against `> 0`:

- [IndexOf](https://learn.microsoft.com/en-us/dotnet/api/system.collections.ilist.indexof), applied to `string`, list or
  array
- [IndexOfAny](https://learn.microsoft.com/en-us/dotnet/api/system.string.indexofany), applied to a `string`
- [LastIndexOf](https://learn.microsoft.com/en-us/dotnet/api/system.string.lastindexof), applied to a `string`, list or
  array
- [LastIndexOfAny](https://learn.microsoft.com/en-us/dotnet/api/system.string.lastindexofany), applied to a `string`

    someArray.IndexOf(someItem) > 0        // Noncompliant: index 0 missing
    someString.IndexOfAny(charsArray) > 0  // Noncompliant: index 0 missing
    someList.LastIndexOf(someItem) > 0     // Noncompliant: index 0 missing
    someString.LastIndexOf(charsArray) > 0 // Noncompliant: index 0 missing

## How to fix it

### Code examples

#### Noncompliant code example

    string color = "blue";
    string name = "ishmael";
    
    List<string> strings = new List<string>();
    strings.Add(color);
    strings.Add(name);
    string[] stringArray = strings.ToArray();
    
    if (strings.IndexOf(color) > 0) // Noncompliant
    {
      // ...
    }
    
    if (name.IndexOf("ish") > 0) // Noncompliant
    {
      // ...
    }
    
    if (name.IndexOf("ae") > 0) // Noncompliant
    {
      // ...
    }
    
    if (Array.IndexOf(stringArray, color) > 0) // Noncompliant
    {
      // ...
    }

#### Compliant solution

    string color = "blue";
    string name = "ishmael";
    
    List<string> strings = new List<string>();
    strings.Add(color);
    strings.Add(name);
    string[] stringArray = strings.ToArray();
    
    if (strings.IndexOf(color) > -1)
    {
      // ...
    }
    
    if (name.IndexOf("ish") >= 0)
    {
      // ...
    }
    
    if (name.Contains("ae"))
    {
      // ...
    }
    
    if (Array.IndexOf(stringArray, color) >= 0)
    {
      // ...
    }

## Resources

### Documentation

- [String.Contains Method](https://learn.microsoft.com/en-us/dotnet/api/system.string.contains)
- [String.IndexOfAny Method](https://learn.microsoft.com/en-us/dotnet/api/system.string.indexofany)
- [String.LastIndexOf Method](https://learn.microsoft.com/en-us/dotnet/api/system.string.lastindexof)
- [String.LastIndexOfAny Method](https://learn.microsoft.com/en-us/dotnet/api/system.string.lastindexofany)