In Unix, "others" class refers to all users except the owner of the file and the members of the group assigned to this file.

In Windows, "Everyone" group is similar and includes all members of the Authenticated Users group as well as the built-in Guest account, and
several other built-in security accounts.

Granting permissions to these groups can lead to unintended access to files.

## Ask Yourself Whether

- The application is designed to be run on a multi-user environment.
- Corresponding files and directories may contain confidential information.

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

The most restrictive possible permissions should be assigned to files and directories.

## Sensitive Code Example

.Net Framework:

    var unsafeAccessRule = new FileSystemAccessRule("Everyone", FileSystemRights.FullControl, AccessControlType.Allow);
    
    var fileSecurity = File.GetAccessControl("path");
    fileSecurity.AddAccessRule(unsafeAccessRule); // Sensitive
    fileSecurity.SetAccessRule(unsafeAccessRule); // Sensitive
    File.SetAccessControl("fileName", fileSecurity);

.Net / .Net Core

    var fileInfo = new FileInfo("path");
    var fileSecurity = fileInfo.GetAccessControl();
    
    fileSecurity.AddAccessRule(new FileSystemAccessRule("Everyone", FileSystemRights.Write, AccessControlType.Allow)); // Sensitive
    fileInfo.SetAccessControl(fileSecurity);

.Net / .Net Core using Mono.Posix.NETStandard

    var fileSystemEntry = UnixFileSystemInfo.GetFileSystemEntry("path");
    fileSystemEntry.FileAccessPermissions = FileAccessPermissions.OtherReadWriteExecute; // Sensitive

## Compliant Solution

.Net Framework

    var safeAccessRule = new FileSystemAccessRule("Everyone", FileSystemRights.FullControl, AccessControlType.Deny);
    
    var fileSecurity = File.GetAccessControl("path");
    fileSecurity.AddAccessRule(safeAccessRule);
    File.SetAccessControl("path", fileSecurity);

.Net / .Net Core

    var safeAccessRule = new FileSystemAccessRule("Everyone", FileSystemRights.FullControl, AccessControlType.Deny);
    
    var fileInfo = new FileInfo("path");
    var fileSecurity = fileInfo.GetAccessControl();
    fileSecurity.SetAccessRule(safeAccessRule);
    fileInfo.SetAccessControl(fileSecurity);

.Net / .Net Core using Mono.Posix.NETStandard

    var fs = UnixFileSystemInfo.GetFileSystemEntry("path");
    fs.FileAccessPermissions = FileAccessPermissions.UserExecute;

## See

- OWASP - [Top 10 2021 Category A1 - Broken Access Control](https://owasp.org/Top10/A01_2021-Broken_Access_Control/)
- OWASP - [Top 10 2021 Category A4 - Insecure Design](https://owasp.org/Top10/A04_2021-Insecure_Design/)
- OWASP - [Top 10 2017 Category A5 - Broken Access Control](https://owasp.org/www-project-top-ten/2017/A5_2017-Broken_Access_Control)
- [OWASP File Permission](https://owasp.org/www-project-web-security-testing-guide/latest/4-Web_Application_Security_Testing/02-Configuration_and_Deployment_Management_Testing/09-Test_File_Permission)
- CWE - [CWE-732 - Incorrect Permission Assignment for Critical Resource](https://cwe.mitre.org/data/definitions/732)
- CWE - [CWE-266 - Incorrect Privilege Assignment](https://cwe.mitre.org/data/definitions/266)
- STIG Viewer - [Application Security and
  Development: V-222430](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222430) - The application must execute without excessive account permissions.