Gratuitous boolean expressions are conditions that do not change the evaluation of a program. This issue can indicate logical errors and affect the
correctness of an application, as well as its maintainability.

## Why is this an issue?

Control flow constructs like `if`-statements allow the programmer to direct the flow of a program depending on a boolean expression.
However, if the condition is always true or always false, only one of the branches will ever be executed. In that case, the control flow construct and
the condition no longer serve a purpose; they become *gratuitous*.

### What is the potential impact?

The presence of gratuitous conditions can indicate a logical error. For example, the programmer *intended* to have the program branch into
different paths but made a mistake when formulating the branching condition. In this case, this issue might result in a bug and thus affect the
reliability of the application. For instance, it might lead to the computation of incorrect results.

Additionally, gratuitous conditions and control flow constructs introduce unnecessary complexity. The source code becomes harder to understand, and
thus, the application becomes more difficult to maintain.

This rule looks for operands of a boolean expression never changing the result of the expression. It also applies to the [null coalescing operator](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/null-coalescing-operator) when one of
the operands always evaluates to `null`.

    string d = null;
    var v1 = d ?? "value";      // Noncompliant

### Exceptions

This rule will not raise an issue in either of these cases:

- When the condition is a single `const bool` 

        const bool debug = false;
        //...
        if (debug)                  // Compliant
        {
          // Print something
        }
- When the condition is the literal `true` or `false`.

In these cases, it is obvious the code is as intended.

## How to fix it

Gratuitous boolean expressions are suspicious and should be carefully removed from the code.

First, the boolean expression in question should be closely inspected for logical errors. If a mistake was made, it can be corrected so the
condition is no longer gratuitous.

If it becomes apparent that the condition is actually unnecessary, it can be removed. The associated control flow construct (e.g., the
`if`-statement containing the condition) will be adapted or even removed, leaving only the necessary branches.

### Code examples

#### Noncompliant code example

    public void Sample(bool b, bool c)
    {
        var a = true;
        if (a)                  // Noncompliant: "a" is always "true"
        {
            DoSomething();
        }
    
        if (b && a)             // Noncompliant: "a" is always "true"
        {
            DoSomething();
        }
    
        if (c || !a)            // Noncompliant: "!a" is always "false"
        {
            DoSomething();
        }
    
        string d = null;
        var v1 = d ?? "value";  // Noncompliant: "d" is always null and v1 is always "value".
        var v2 = s ?? d;        // Noncompliant: "d" is always null and v2 is always equal to s.
    }

#### Compliant solution

The unnecessary operand is updated:

    public void Sample(bool b, bool c, string s)
    {
        var a = IsAllowed();
        if (a)                  // Compliant
        {
            DoSomething();
        }
    
        if (b && a)             // Compliant
        {
            DoSomething();
        }
    
        if (c || !a)            // Compliant
        {
            DoSomething();
        }
    
        string d = GetStringData();
        var v1 = d ?? "value";  // Compliant
        var v2 = s ?? d;        // Compliant
    }

The unnecessary operand is removed:

    public void Sample(bool b, bool c, string s)
    {
        DoSomething();
    
        if (b)                  // Compliant
        {
            DoSomething();
        }
    
        if (c)                  // Compliant
        {
            DoSomething();
        }
    
        var v1 = "value";       // Compliant
        var v2 = s;             // Compliant
    }

## Resources

### Documentation

- CWE - [CWE-571 - Expression is Always True](https://cwe.mitre.org/data/definitions/571)
- CWE - [CWE-570 - Expression is Always False](https://cwe.mitre.org/data/definitions/570)
- [Conditional logical AND operator &&](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/boolean-logical-operators#conditional-logical-and-operator-)
- [Conditional logical OR operator ||](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/boolean-logical-operators#conditional-logical-or-operator-)
- [Null-coalescing operators ??
  and ??=](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/null-coalescing-operator)