## Why is this an issue?

Conditional expressions which are always `true` or `false` can lead to [unreachable code](https://en.wikipedia.org/wiki/Unreachable_code).

In the case below, the call of `Dispose()` never happens.

    var a = false;
    if (a)
    {
        Dispose(); // Never reached
    }

### Exceptions

This rule will not raise an issue in either of these cases:

- When the condition is a single `const bool` 

        const bool debug = false;
        //...
        if (debug)
        {
          // Print something
        }
- When the condition is the literal `true` or `false`.

In these cases, it is obvious the code is as intended.

## How to fix it

The conditions should be reviewed to decide whether:

- to update the condition or
- to remove the condition.

### Code examples

#### Noncompliant code example

    public void Sample(bool b)
    {
        bool a = false;
        if (a)                  // Noncompliant: The true branch is never reached
        {
            DoSomething();      // Never reached
        }
    
        if (!a || b)            // Noncompliant: "!a" is always "true" and the false branch is never reached
        {
            DoSomething();
        }
        else
        {
            DoSomethingElse();  // Never reached
        }
    
        var c = "xxx";
        var res = c ?? "value"; // Noncompliant: c is always not null, "value" is never used
    }

#### Compliant solution

    public void Sample(bool b)
    {
        bool a = false;
        if (Foo(a))             // Condition was updated
        {
            DoSomething();
        }
    
        if (b)                  // Parts of the condition were removed.
        {
            DoSomething();
        }
        else
        {
            DoSomethingElse();
        }
    
        var c = "xxx";
        var res = c;            // ?? "value" was removed
    }

## Resources

- CWE - [CWE-570 - Expression is Always False](https://cwe.mitre.org/data/definitions/570)
- CWE - [CWE-571 - Expression is Always True](https://cwe.mitre.org/data/definitions/571)
- Wikipedia - [Unreachable code](https://en.wikipedia.org/wiki/Unreachable_code)

### Documentation

- Microsoft Learn - [Conditional logical AND operator &&](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/boolean-logical-operators#conditional-logical-and-operator-)
- Microsoft Learn - [Conditional logical OR operator ||](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/boolean-logical-operators#conditional-logical-or-operator-)
- Microsoft Learn - [?? and ??=
  operators - the null-coalescing operators](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/operators/null-coalescing-operator)