## Why is this an issue?

A shared resource refers to a resource or data that can be accessed or modified by multiple [threads](https://en.wikipedia.org/wiki/Thread_%28computing%29) or concurrent parts of a program. It could be any piece of data, object, file,
database connection, or system resource that needs to be accessed or manipulated by multiple parts of a program at the same time.

Shared resources should not be used for [locking](https://en.wikipedia.org/wiki/Lock_%28computer_science%29) because it increases the chance
of [deadlocks](https://en.wikipedia.org/wiki/Deadlock). Any other thread could acquire (or attempt to acquire) the same lock while doing
some operation, without knowing that the resource is meant to be used for locking purposes.

For example, a `string` should never be used for locking. When a `string` is [interned](https://en.wikipedia.org/wiki/Interning_%28computer_science%29) by the runtime, it can be shared by multiple threads, breaking the
locking mechanism.

Instead, a dedicated private `object` instance should be used for each shared resource. This minimizes access to the lock instance,
avoiding deadlocks and lock contention.

The following objects are considered as shared resources:

- a reference to [this](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/this): if the instance is publicly
  accessibly, the lock might be shared
- a [Type](https://learn.microsoft.com/en-us/dotnet/api/system.type) object: if the type class is publicly accessibly, the lock might
  be shared
- a [string](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/strings/) literal or instance: if any other part of the
  program uses the same string, the lock is shared because of interning

## How to fix it

### Code examples

#### Noncompliant code example

    void MyLockingMethod()
    {
        lock (this) // Noncompliant
        {
            // ...
        }
    }

#### Compliant solution

    private readonly object lockObj = new object();
    void MyLockingMethod()
    {
        lock (lockObj)
        {
            // ...
        }
    }

## Resources

### Documentation

- [Thread](https://en.wikipedia.org/wiki/Thread_%28computing%29)
- [Locking](https://en.wikipedia.org/wiki/Lock_%28computer_science%29)
- [Deadlock](https://en.wikipedia.org/wiki/Deadlock)
- [Interning](https://en.wikipedia.org/wiki/Interning_%28computer_science%29)
- [String interning by the runtime](https://learn.microsoft.com/en-us/dotnet/api/system.string.intern#remarks)
- [Managed Threading Best Practices](https://docs.microsoft.com/en-us/dotnet/standard/threading/managed-threading-best-practices)