## Why is this an issue?

[Locking](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/lock) on a class field synchronizes not on the
field itself, but on the object assigned to it. Thus, there are some good practices to follow to avoid problems related to [thread](https://learn.microsoft.com/en-us/dotnet/standard/threading/threads-and-threading) synchronization.

- Locking on a non-`readonly` field makes it possible for the field’s value to change while a thread is in the code block, locked on the old value. This allows another thread to lock on the new value and access the same block concurrently. 

        private Color color = new Color("red");
        private void DoSomething()
        {
          // Synchronizing access via "color"
          lock (color) // Noncompliant: lock is actually on object instance "red" referred to by the "color" field
          {
            //...
            color = new Color("green"); // other threads now allowed into this block
            // ...
          }
        }
- Locking on a new instance of an object undermines synchronization because two different threads running the same method in parallel will lock on different instances of the same object, allowing them to access the synchronized block at the same time. 

        private void DoSomething()
        {
          lock (new object()) // Noncompliant: every thread locks on a different new instance
          {
            // ...
          }
        }
- Locking on a string literal is also dangerous since, depending on whether the string is interned or not, different threads may or may not synchronize on the same object instance. 

        private readonly string colorString = "red";
        private void DoSomething()
        {
          lock (colorString)  // Noncompliant: strings can be interned
          {
            // ...
          }
        }

## How to fix it

### Code examples

#### Noncompliant code example

    private Color color = new Color("red");
    private void DoSomething()
    {
      // Synchronizing access via "color"
      lock (color) // Noncompliant: lock is actually on object instance "red" referred to by the "color" field
      {
        //...
        color = new Color("green"); // other threads now allowed into this block
        // ...
      }
    }

#### Compliant solution

    private Color color = new Color("red");
    private readonly object lockObj = new object();
    
    private void DoSomething()
    {
      lock (lockObj)
      {
        //...
        color = new Color("green");
        // ...
      }
    }

## Resources

- [Lock Statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/lock) - lock statement - ensure
  exclusive access to a shared resource
- [String.Intern](https://learn.microsoft.com/en-us/dotnet/api/system.string.intern) - `String.Intern(String)` Method
- CWE - [CWE-412 - Unrestricted Externally Accessible Lock](https://cwe.mitre.org/data/definitions/412)
- CWE - [CWE-413 - Improper Resource Locking](https://cwe.mitre.org/data/definitions/413)
- [Threads and threading](https://learn.microsoft.com/en-us/dotnet/standard/threading/threads-and-threading)