## Why is this an issue?

Using [multidimensional](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/multidimensional-arrays) and [jagged](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/jagged-arrays) arrays as method parameters in C# can be
challenging for developers.

When these methods are exposed to external users, it requires advanced language knowledge for effective usage.

Determining the appropriate data to pass to these parameters may not be intuitive.

    public class Program
    {
        public void WriteMatrix(int[][] matrix) // Noncompliant: data type is not intuitive
        {
            // ...
        }
    }

In this example, it cannot be inferred easily what the matrix should look like. Is it a 2x2 Matrix or even a triangular Matrix?

Using a collection, data structure, or class that provides a more suitable representation of the required data is recommended instead of a
multidimensional array or jagged array to enhance code readability.

    public class Matrix2x2
    {
        // ...
    }
    
    public class Program
    {
        public void WriteMatrix(Matrix2x2 matrix) // Compliant: data type is intuitive
        {
            // ...
        }
    }

As a result, avoiding exposing such methods to external users is recommended.

### Exceptions

However, using multidimensional and jagged array method parameters internally, such as in `private` or `internal` methods or
within `internal` classes, is compliant since they are not publicly exposed.

    public class FirstClass
    {
        private void UpdateMatrix(int[][] matrix) // Compliant: method is private
        {
            // ...
        }
    }
    
    internal class SecondClass
    {
        public void UpdateMatrix(int[][] matrix) // Compliant: class is internal
        {
            // ...
        }
    }

## Resources

### Documentation

- [Collections and Data Structures](https://learn.microsoft.com/en-us/dotnet/standard/collections/)
- [Jagged Arrays](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/jagged-arrays)
- [Multidimensional Arrays](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/arrays/multidimensional-arrays)