## Why is this an issue?

[Enumerable.Sum()](https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.sum) always executes addition in a
`checked` context, so an [OverflowException](https://learn.microsoft.com/en-us/dotnet/api/system.overflowexception) will be
thrown if the value exceeds `MaxValue`, even if an `unchecked` context was specified. Therefore, using this method inside an
`unchecked` context will only make the code more confusing, since the behavior will still be `checked`.

This rule raises an issue when an `unchecked` context is specified for a `Sum` on integer types.

### Exceptions

When the `Sum` call is inside a [try-catch block](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/exceptions/),
no issues are reported, since the exception is properly handled.

    void Add(List<int> list)
    {
      unchecked
      {
        try
        {
          int total = list.Sum();
        }
        catch (System.OverflowException e)
        {
          // Exception handling
        }
      }
    }

## How to fix it

Remove the `unchecked` operator/statement, and optionally add some exception handling for the `OverflowException`.

### Code examples

#### Noncompliant code example

    void Add(List<int> list)
    {
      int total1 = unchecked(list.Sum());  // Noncompliant
    
      unchecked
      {
        int total2 = list.Sum();  // Noncompliant
      }
    }

#### Compliant solution

    void Add(List<int> list)
    {
      int total1 = list.Sum();
    
      try
      {
        int total2 = list.Sum();
      }
      catch (System.OverflowException e)
      {
        // Exception handling
      }
    }

## Resources

### Documentation

- [`Enumerable.Sum` Method](https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.sum)
- [`Enumerable.Sum` implementation](https://github.com/microsoft/referencesource/blob/51cf7850defa8a17d815b4700b67116e3fa283c2/System.Core/System/Linq/Enumerable.cs#L1408-L1415)
- [`checked` and
  `unchecked` statements](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/checked-and-unchecked)
- [`checked` and `unchecked` operators](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/language-specification/expressions#12819-the-checked-and-unchecked-operators)
- [Exceptions and Exception Handling](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/exceptions/)
- [`OverflowException` Class](https://learn.microsoft.com/en-us/dotnet/api/system.overflowexception)
- [Integer overflow](https://en.wikipedia.org/wiki/Integer_overflow)