The use of a non-standard algorithm is dangerous because a determined attacker may be able to break the algorithm and compromise whatever data has
been protected. Standard algorithms like `AES`, `RSA`, `SHA`, …​ should be used instead.

This rule tracks custom implementation of these types from `System.Security.Cryptography` namespace:

- `AsymmetricAlgorithm`
- `AsymmetricKeyExchangeDeformatter`
- `AsymmetricKeyExchangeFormatter`
- `AsymmetricSignatureDeformatter`
- `AsymmetricSignatureFormatter`
- `DeriveBytes`
- `HashAlgorithm`
- `ICryptoTransform`
- `SymmetricAlgorithm`

## Recommended Secure Coding Practices

- Use a standard algorithm instead of creating a custom one.

## Sensitive Code Example

    public class CustomHash : HashAlgorithm // Noncompliant
    {
        private byte[] result;
    
        public override void Initialize() => result = null;
        protected override byte[] HashFinal() => result;
    
        protected override void HashCore(byte[] array, int ibStart, int cbSize) =>
            result ??= array.Take(8).ToArray();
    }

## Compliant Solution

    SHA256 mySHA256 = SHA256.Create()

## See

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- CWE - [CWE-327 - Use of a Broken or Risky Cryptographic Algorithm](https://cwe.mitre.org/data/definitions/327)
- Derived from FindSecBugs rule [MessageDigest is Custom](https://h3xstream.github.io/find-sec-bugs/bugs.htm#CUSTOM_MESSAGE_DIGEST)