Using pseudorandom number generators (PRNGs) is security-sensitive. For example, it has led in the past to the following vulnerabilities:

- [CVE-2013-6386](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2013-6386)
- [CVE-2006-3419](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2006-3419)
- [CVE-2008-4102](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2008-4102)

When software generates predictable values in a context requiring unpredictability, it may be possible for an attacker to guess the next value that
will be generated, and use this guess to impersonate another user or access sensitive information.

As the `System.Random` class relies on a pseudorandom number generator, it should not be used for security-critical applications or for
protecting sensitive data. In such context, the `System.Cryptography.RandomNumberGenerator` class which relies on a cryptographically
strong random number generator (RNG) should be used in place.

## Ask Yourself Whether

- the code using the generated value requires it to be unpredictable. It is the case for all encryption mechanisms or when a secret value, such
  as a password, is hashed.
- the function you use generates a value which can be predicted (pseudo-random).
- the generated value is used multiple times.
- an attacker can access the generated value.

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- Only use random number generators which are [recommended by
  OWASP](https://cheatsheetseries.owasp.org/cheatsheets/Cryptographic_Storage_Cheat_Sheet.html#secure-random-number-generation) or any other trusted organization.
- Use the generated random values only once.
- You should not expose the generated random value. If you have to store it, make sure that the database or file is secure.

## Sensitive Code Example

    var random = new Random(); // Sensitive use of Random
    byte[] data = new byte[16];
    random.NextBytes(data);
    return BitConverter.ToString(data); // Check if this value is used for hashing or encryption

## Compliant Solution

    using System.Security.Cryptography;
    ...
    var randomGenerator = RandomNumberGenerator.Create(); // Compliant for security-sensitive use cases
    byte[] data = new byte[16];
    randomGenerator.GetBytes(data);
    return BitConverter.ToString(data);

## See

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- [Mobile AppSec Verification Standard - Cryptography Requirements](https://mas.owasp.org/checklists/MASVS-CRYPTO/)
- OWASP - [Mobile Top 10 2016 Category M5 -
  Insufficient Cryptography](https://owasp.org/www-project-mobile-top-10/2016-risks/m5-insufficient-cryptography)
- CWE - [CWE-338 - Use of Cryptographically Weak Pseudo-Random Number Generator (PRNG)](https://cwe.mitre.org/data/definitions/338)
- CWE - [CWE-330 - Use of Insufficiently Random Values](https://cwe.mitre.org/data/definitions/330)
- CWE - [CWE-326 - Inadequate Encryption Strength](https://cwe.mitre.org/data/definitions/326)
- CWE - [CWE-1241 - Use of Predictable Algorithm in Random Number Generator](https://cwe.mitre.org/data/definitions/1241)
- Derived from FindSecBugs rule [Predictable Pseudo Random Number
  Generator](https://h3xstream.github.io/find-sec-bugs/bugs.htm#PREDICTABLE_RANDOM)