## Why is this an issue?

To prevent potential [deadlocks](https://en.wikipedia.org/wiki/Deadlock) in an application, it is crucial to release any locks that are
acquired within a method along all possible execution paths.

Failing to release locks properly can lead to potential deadlocks, where the lock might not be released, causing issues in the application.

This rule specifically focuses on tracking the following types from the `System.Threading` namespace:

- [`Monitor`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.monitor)
- [`Mutex`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.mutex)
- [`ReaderWriterLock`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.readerwriterlock)
- [`ReaderWriterLockSlim`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.readerwriterlockslim)
- [`SpinLock`](https://learn.microsoft.com/en-us/dotnet/api/system.threading.spinlock)

An issue is reported when a lock is acquired within a method but not released on all paths.

### Exceptions

If the lock is never released within the method, no issue is raised, assuming that the callers will handle the release.

## How to fix it

To make sure that a lock is always released correctly, you can follow one of these two methods:

- Use a [`lock`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/lock) statement with your
  lock object.
- Use a [`try-finally`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements#the-try-finally-statement) statement and put the release of your lock object within the finally block.

### Code examples

#### Noncompliant code example

    class MyClass
    {
      private object obj = new object();
    
      public void DoSomethingWithMonitor()
      {
        Monitor.Enter(obj); // Noncompliant: not all paths release the lock
        if (IsInitialized())
        {
          // ...
          Monitor.Exit(obj);
        }
      }
    }

    class MyClass
    {
      private ReaderWriterLockSlim lockObj = new ReaderWriterLockSlim();
    
      public void DoSomethingWithReaderWriteLockSlim()
      {
        lockObj.EnterReadLock(); // Noncompliant: not all paths release the lock
        if (IsInitialized())
        {
          // ...
          lockObj.ExitReadLock();
        }
      }
    }

#### Compliant solution

    class MyClass
    {
      private object obj = new object();
    
      public void DoSomethingWithMonitor()
      {
        lock(obj) // Compliant: the lock will be released at the end of the lock block
        {
          if (IsInitialized())
          {
            // ...
          }
        }
      }
    }

    class MyClass
    {
      private ReaderWriterLockSlim lockObj = new ReaderWriterLockSlim();
    
      public void DoSomethingWithReaderWriteLockSlim()
      {
        lockObj.EnterReadLock(); // Compliant: the lock will be released in the finally block
        try
        {
          if (IsInitialized())
          {
            // ...
          }
        }
        finally
        {
          lockObj.ExitReadLock();
        }
      }
    }

## Resources

- [Synchronization of access to a shared resource](https://docs.microsoft.com/en-us/dotnet/standard/threading/overview-of-synchronization-primitives#synchronization-of-access-to-a-shared-resource)
- CWE - [CWE-459 - Incomplete Cleanup](https://cwe.mitre.org/data/definitions/459)
- [`lock` statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/lock)
- [The
  `try-finally` statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements#the-try-finally-statement)