## Why is this an issue?

When you do not use the return value of a method with no side effects, it indicates that something is wrong. Either this method is unnecessary, or
the source code does not behave as expected and could lead to code defects. For example, there are methods, such as [DateTime.AddYears](https://learn.microsoft.com/en-us/dotnet/api/system.datetime.addyears), that don’t change the value of the input object,
but instead, they return a new object whose value is the result of this operation, and as a result that you will have unexpected effects if you do not
use the return value.

This rule raises an issue when the results of the following methods are ignored:

- [LINQ](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/concepts/linq/)
- [`Pure` methods](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.contracts.pureattribute)
- Any method on [build-in types](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/built-in-types)
- Any method on [Immutable
  collections](https://learn.microsoft.com/en-us/archive/msdn-magazine/2017/march/net-framework-immutable-collections)

Special cases:

- Although [`string.Intern`](https://learn.microsoft.com/en-us/dotnet/api/system.string.intern) has a side effect, ignoring
  its return value is still suspicious as it is the only reference ensured to point to the intern pool.
- LINQ methods can have side effects if they are misused. For example:

    data.All(x =>
    {
        x.Property = "foo";
        return true;
    });

Such code should be rewritten as a loop because [`Enumerable.All<TSource>`](https://learn.microsoft.com/en-us/dotnet/api/system.linq.enumerable.all) method should be used to
determine if all elements satisfy a condition and not to change their state.

### Exceptions

This rule doesn’t report issues on invocations with [`out`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/out-parameter-modifier) or [`ref`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/ref) arguments.

## How to fix it

### Code examples

#### Noncompliant code example

    data.Where(x => x > 5).Select(x => x * x); // Noncompliant
    "this string".Equals("other string"); // Noncompliant
    
    data.All(x =>  // Noncompliant
    {
        x.Property = "foo";
        return true;
    });

#### Compliant solution

    var res = data.Where(x => x > 5).Select(x => x * x);
    var isEqual = "this string".Equals("other string");
    
    foreach (var x in data)
    {
        x.Property = "foo";
    }

## Resources

### Documentation

- Microsoft Learn - [`PureAttribute`
  Class](https://learn.microsoft.com/en-us/dotnet/api/system.diagnostics.contracts.pureattribute)
- Microsoft Learn - [`out`
  parameter modifier](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/out-parameter-modifier)
- Microsoft Learn - [`ref` keyword](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/keywords/ref)
- Microsoft Learn - [`String.Intern(String)` Method](https://learn.microsoft.com/en-us/dotnet/api/system.string.intern)
- Microsoft Learn - [LINQ](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/concepts/linq/)
- Microsoft Learn - [build-in types](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/builtin-types/built-in-types)
- Microsoft Learn - [Immutable
  collections](https://learn.microsoft.com/en-us/archive/msdn-magazine/2017/march/net-framework-immutable-collections)

### Articles & blog posts

- [Don’t ignore your functions](https://www.daniellittle.dev/dont-ignore-your-functions)