## Why is this an issue?

To ensure proper testing, it is important to include test cases in a test class. If a test class does not have any test cases, it can give the
wrong impression that the class being tested has been thoroughly tested, when in reality, it has not.

This rule will raise an issue when any of these conditions are met:

- For `NUnit`, a class is marked with `TestFixture` but does not contain any method marked with `Test`,
  `TestCase`, `TestCaseSource`, or `Theory`.
- For `MSTest`, a class is marked with `TestClass` but does not contain any method marked with `TestMethod` or
  `DataTestMethod`.

It does not apply to `xUnit` since `xUnit` does not require a [test
class attribute](https://xunit.net/docs/comparisons#attributes).

### Exceptions

There are scenarios where not having any test cases within a test class is perfectly acceptable and not seen as a problem.

#### Abstract classes

To facilitate the creation of common test cases, test logic, or test infrastructure, it is advisable to use a base class.

Additionally, in both `NUnit` and `MSTest`, abstract classes that are annotated with their respective attributes
(`TestFixture` in NUnit and `TestClass` in MSTest) are automatically ignored.

Therefore, there is no need to raise an issue in this particular scenario.

More information here:

- [`TestFixture` documentation in
  `NUnit`](https://docs.nunit.org/articles/nunit/writing-tests/attributes/testfixture.html)
- [`TypeValidator` class in `MSTest` (GitHub)](https://github.com/microsoft/testfx/blob/0f19160cc319338ef6e23acb320da1562b40decd/src/Adapter/MSTest.TestAdapter/Discovery/TypeValidator.cs#L86-L97)

#### Derived classes that inherit test cases from a base class

A base class containing one or more test cases to provide generic test cases is also considered a compliant scenario.

#### Classes that contain `AssemblyInitialize` or `AssemblyCleanup` methods

**This particular exception scenario only applies to the MSTest test framework.**

The `AssemblyInitialize` and `AssemblyCleanup` attributes are used to annotate methods that are executed only once at the
beginning and at the end of a test run. These attributes can only be applied once per assembly.

It is logical to have a dedicated class for these methods, and this scenario is also considered compliant.

Furthermore, it is important to note that the test engine will execute a method annotated with either the `AssemblyInitialize` or
`AssemblyCleanup` attribute only if that method is part of a class annotated with the `TestClass` attribute.

More information here:

- [`AssemblyInitialize`
  attribute](https://learn.microsoft.com/en-us/previous-versions/visualstudio/visual-studio-2008/ms245278%28v=vs.90%29)
- [`AssemblyCleanup`
  attribute](https://learn.microsoft.com/en-us/previous-versions/visualstudio/visual-studio-2008/ms245265%28v=vs.90%29)

## How to fix it in MSTest

To fix this issue in `MSTest`, it is important that all test classes annotated with the `[TestClass]` attribute contain at
least one test case.

To achieve this, at least one method needs to be annotated with one of the following method attributes:

- `TestMethod`
- `DataTestMethod`

### Code examples

#### Noncompliant code example

    [TestClass]
    public class SomeOtherClassTest { } // Noncompliant: no test

#### Compliant solution

    [TestClass]
    public class SomeOtherClassTest
    {
        [TestMethod]
        public void SomeMethodShouldReturnTrue() { }
    }

## How to fix it in NUnit

To fix this issue in `NUnit`, it is important that all test classes annotated with the `[TestFixture]` attribute contain at
least one test case.

To achieve this, at least one method needs to be annotated with one of the following method attributes:

- `Test`
- `TestCase`
- `TestCaseSource`
- `Theory`

### Code examples

#### Noncompliant code example

    [TestFixture]
    public class SomeClassTest { } // Noncompliant: no test

#### Compliant solution

    [TestFixture]
    public class SomeClassTest
    {
        [Test]
        public void SomeMethodShouldReturnTrue() { }
    }

## Resources

### Documentation

- [`NUnit` documentation](https://docs.nunit.org/articles/nunit/intro.html)
    - [`TestFixture` attribute](https://docs.nunit.org/articles/nunit/writing-tests/attributes/testfixture.html)
    - [`Test` attribute](https://docs.nunit.org/articles/nunit/writing-tests/attributes/test.html)
    - [`TestCase` attribute](https://docs.nunit.org/articles/nunit/writing-tests/attributes/testcase.html)
    - [`TestCaseSource` attribute](https://docs.nunit.org/articles/nunit/writing-tests/attributes/testcasesource.html)
    - [`Theory` attribute](https://docs.nunit.org/articles/nunit/writing-tests/attributes/theory.html)
- [Unit testing C# with MSTest](https://learn.microsoft.com/en-us/dotnet/core/testing/unit-testing-with-mstest)
- [`MSTest` documentation](https://github.com/microsoft/testfx/blob/main/docs/README.md)
- [Comparing `xUnit` to other frameworks - Attributes](https://xunit.net/docs/comparisons#attributes)