When accessing a database, an empty password should be avoided as it introduces a weakness.

## Why is this an issue?

When a database does not require a password for authentication, it allows anyone to access and manipulate the data stored within it. Exploiting
this vulnerability typically involves identifying the target database and establishing a connection to it without the need for any authentication
credentials.

### What is the potential impact?

Once connected, an attacker can perform various malicious actions, such as viewing, modifying, or deleting sensitive information, potentially
leading to data breaches or unauthorized access to critical systems. It is crucial to address this vulnerability promptly to ensure the security and
integrity of the database and the data it contains.

#### Unauthorized Access to Sensitive Data

When a database lacks a password for authentication, it opens the door for unauthorized individuals to gain access to sensitive data. This can
include personally identifiable information (PII), financial records, intellectual property, or any other confidential information stored in the
database. Without proper access controls in place, malicious actors can exploit this vulnerability to retrieve sensitive data, potentially leading to
identity theft, financial loss, or reputational damage.

#### Compromise of System Integrity

Without a password requirement, unauthorized individuals can gain unrestricted access to a database, potentially compromising the integrity of the
entire system. Attackers can inject malicious code, alter configurations, or manipulate data within the database, leading to system malfunctions,
unauthorized system access, or even complete system compromise. This can disrupt business operations, cause financial losses, and expose the
organization to further security risks.

#### Unwanted Modifications or Deletions

The absence of a password for database access allows anyone to make modifications or deletions to the data stored within it. This poses a
significant risk, as unauthorized changes can lead to data corruption, loss of critical information, or the introduction of malicious content. For
example, an attacker could modify financial records, tamper with customer orders, or delete important files, causing severe disruptions to business
processes and potentially leading to financial and legal consequences.

Overall, the lack of a password configured to access a database poses a serious security risk, enabling unauthorized access, data breaches, system
compromise, and unwanted modifications or deletions. It is essential to address this vulnerability promptly to safeguard sensitive data, maintain
system integrity, and protect the organization from potential harm.

## How to fix it in Entity Framework Core

### Code examples

The following code uses an empty password to connect to a SQL Server database.

The vulnerability can be fixed by using Windows authentication (sometimes referred to as integrated security).

#### Noncompliant code example

    protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
    {
      optionsBuilder.UseSqlServer("Server=myServerAddress;Database=myDataBase;User Id=myUsername;Password="); // Noncompliant
    }

#### Compliant solution

    protected override void OnConfiguring(DbContextOptionsBuilder optionsBuilder)
    {
      optionsBuilder.UseSqlServer("Server=myServerAddress;Database=myDataBase;Integrated Security=True");
    }

### How does this work?

#### Windows authentication (integrated security)

When the connection string includes the `Integrated Security=true` parameter, it enables Windows authentication (sometimes called
integrated security) for the database connection. With integrated security, the user’s Windows credentials are used to authenticate and authorize
access to the database. It eliminates the need for a separate username and password for the database connection. Integrated security simplifies
authentication and leverages the existing Windows authentication infrastructure for secure database access in your C# application.

It’s important to note that when using integrated security, the user running the application must have the necessary permissions to access the
database. Ensure that the user account running the application has the appropriate privileges and is granted access to the database.

The syntax employed in connection strings varies by provider:

| Syntax | Supported by |
| --- | --- |
| `Integrated Security=true;` | SQL Server, Oracle, Postgres |
| `Integrated Security=SSPI;` | SQL Server, OLE DB |
| `Integrated Security=yes;` | MySQL |
| `Trusted_Connection=true;` | SQL Server |
| `Trusted_Connection=yes;` | ODBC |

Note: Some providers such as MySQL do not support Windows authentication with .NET Core.

### Pitfalls

#### Hard-coded passwords

It could be tempting to replace the empty password with a hard-coded one. Hard-coding passwords in the code can pose significant security risks.
Here are a few reasons why it is not recommended:

1. Security Vulnerability: Hard-coded passwords can be easily discovered by anyone who has access to the code, such as other developers or
  attackers. This can lead to unauthorized access to the database and potential data breaches.
2. Lack of Flexibility: Hard-coded passwords make it difficult to change the password without modifying the code. If the password needs to be
  updated, it would require recompiling and redeploying the code, which can be time-consuming and error-prone.
3. Version Control Issues: Storing passwords in code can lead to version control issues. If the code is shared or stored in a version control
  system, the password will be visible to anyone with access to the repository, which is a security risk.

To mitigate these risks, it is recommended to use secure methods for storing and retrieving passwords, such as using environment variables,
configuration files, or secure key management systems. These methods allow for better security, flexibility, and separation of sensitive information
from the codebase.

## How to fix it in ASP.NET

### Code examples

The following configuration file uses an empty password to connect to a database.

The vulnerability can be fixed by using Windows authentication (sometimes referred to as integrated security)

#### Noncompliant code example

    <?xml version="1.0" encoding="utf-8"?>
    <configuration>
      <connectionStrings>
        <add name="myConnection" connectionString="Server=myServerAddress;Database=myDataBase;User Id=myUsername;Password=" /> <!-- Noncompliant -->
      </connectionStrings>
    </configuration>

#### Compliant solution

    <?xml version="1.0" encoding="utf-8"?>
    <configuration>
      <connectionStrings>
        <add name="myConnection" connectionString="Server=myServerAddress;Database=myDataBase;Integrated Security=True" />
      </connectionStrings>
    </configuration>

### How does this work?

#### Windows authentication (integrated security)

When the connection string includes the `Integrated Security=true` parameter, it enables Windows authentication (sometimes called
integrated security) for the database connection. With integrated security, the user’s Windows credentials are used to authenticate and authorize
access to the database. It eliminates the need for a separate username and password for the database connection. Integrated security simplifies
authentication and leverages the existing Windows authentication infrastructure for secure database access in your C# application.

It’s important to note that when using integrated security, the user running the application must have the necessary permissions to access the
database. Ensure that the user account running the application has the appropriate privileges and is granted access to the database.

The syntax employed in connection strings varies by provider:

| Syntax | Supported by |
| --- | --- |
| `Integrated Security=true;` | SQL Server, Oracle, Postgres |
| `Integrated Security=SSPI;` | SQL Server, OLE DB |
| `Integrated Security=yes;` | MySQL |
| `Trusted_Connection=true;` | SQL Server |
| `Trusted_Connection=yes;` | ODBC |

Note: Some providers such as MySQL do not support Windows authentication with .NET Core.

### Pitfalls

#### Hard-coded passwords

It could be tempting to replace the empty password with a hard-coded one. Hard-coding passwords in the code can pose significant security risks.
Here are a few reasons why it is not recommended:

1. Security Vulnerability: Hard-coded passwords can be easily discovered by anyone who has access to the code, such as other developers or
  attackers. This can lead to unauthorized access to the database and potential data breaches.
2. Lack of Flexibility: Hard-coded passwords make it difficult to change the password without modifying the code. If the password needs to be
  updated, it would require recompiling and redeploying the code, which can be time-consuming and error-prone.
3. Version Control Issues: Storing passwords in code can lead to version control issues. If the code is shared or stored in a version control
  system, the password will be visible to anyone with access to the repository, which is a security risk.

To mitigate these risks, it is recommended to use secure methods for storing and retrieving passwords, such as using environment variables,
configuration files, or secure key management systems. These methods allow for better security, flexibility, and separation of sensitive information
from the codebase.

## Resources

- [Create the Web.config file for an ASP.NET application](https://docs.microsoft.com/en-us/troubleshoot/aspnet/create-web-config)

### Standards

- OWASP - [Top 10 2021 Category A7 - Identification and
  Authentication Failures](https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/)
- OWASP - [Top 10 2017 Category A2 - Broken Authentication](https://owasp.org/www-project-top-ten/2017/A2_2017-Broken_Authentication)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- CWE - [CWE-521 - Weak Password Requirements](https://cwe.mitre.org/data/definitions/521)