Because it is easy to extract strings from an application source code or binary, credentials should not be hard-coded. This is particularly true
for applications that are distributed or that are open-source.

In the past, it has led to the following vulnerabilities:

- [CVE-2019-13466](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2019-13466)
- [CVE-2018-15389](http://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2018-15389)

Credentials should be stored outside of the code in a configuration file, a database, or a management service for secrets.

This rule flags instances of hard-coded credentials used in database and LDAP connections. It looks for hard-coded credentials in connection
strings, and for variable names that match any of the patterns from the provided list.

It’s recommended to customize the configuration of this rule with additional credential words such as "oauthToken", "secret", …​

## Ask Yourself Whether

- Credentials allow access to a sensitive component like a database, a file storage, an API or a service.
- Credentials are used in production environments.
- Application re-distribution is required before updating the credentials.

There is a risk if you answered yes to any of those questions.

## Recommended Secure Coding Practices

- Store the credentials in a configuration file that is not pushed to the code repository.
- Store the credentials in a database.
- Use your cloud provider’s service for managing secrets.
- If a password has been disclosed through the source code: change it.

## Sensitive Code Example

    string username = "admin";
    string password = "Admin123"; // Sensitive
    string usernamePassword  = "user=admin&password=Admin123"; // Sensitive
    string url = "scheme://user:Admin123@domain.com"; // Sensitive

## Compliant Solution

    string username = "admin";
    string password = GetEncryptedPassword();
    string usernamePassword = string.Format("user={0}&password={1}", GetEncryptedUsername(), GetEncryptedPassword());
    string url = $"scheme://{username}:{password}@domain.com";
    
    string url2 = "http://guest:guest@domain.com"; // Compliant
    const string Password_Property = "custom.password"; // Compliant

## Exceptions

- Issue is not raised when URI username and password are the same.
- Issue is not raised when searched pattern is found in variable name and value.

## See

- OWASP - [Top 10 2021 Category A7 - Identification and
  Authentication Failures](https://owasp.org/Top10/A07_2021-Identification_and_Authentication_Failures/)
- OWASP - [Top 10 2017 Category A2 - Broken Authentication](https://owasp.org/www-project-top-ten/2017/A2_2017-Broken_Authentication)
- CWE - [CWE-798 - Use of Hard-coded Credentials](https://cwe.mitre.org/data/definitions/798)
- CWE - [CWE-259 - Use of Hard-coded Password](https://cwe.mitre.org/data/definitions/259)
- Derived from FindSecBugs rule [Hard Coded Password](https://h3xstream.github.io/find-sec-bugs/bugs.htm#HARD_CODE_PASSWORD)