This vulnerability increases the likelihood that attackers are able to compute the cleartext of password hashes.

## Why is this an issue?

During the process of password hashing, an additional component, known as a "salt," is often integrated to bolster the overall security. This salt,
acting as a defensive measure, primarily wards off certain types of attacks that leverage pre-computed tables to crack passwords.

However, potential risks emerge when the salt is deemed insecure. This can occur when the salt is consistently the same across all users or when it
is too short or predictable. In scenarios where users share the same password and salt, their password hashes will inevitably mirror each other.
Similarly, a short salt heightens the probability of multiple users unintentionally having identical salts, which can potentially lead to identical
password hashes. These identical hashes streamline the process for potential attackers to recover clear-text passwords. Thus, the emphasis on
implementing secure, unique, and sufficiently lengthy salts in password-hashing functions is vital.

### What is the potential impact?

Despite best efforts, even well-guarded systems might have vulnerabilities that could allow an attacker to gain access to the hashed passwords.
This could be due to software vulnerabilities, insider threats, or even successful phishing attempts that give attackers the access they need.

Once the attacker has these hashes, they will likely attempt to crack them using a couple of methods. One is brute force, which entails trying
every possible combination until the correct password is found. While this can be time-consuming, having the same salt for all users or a short salt
can make the task significantly easier and faster.

If multiple users have the same password and the same salt, their password hashes would be identical. This means that if an attacker successfully
cracks one hash, they have effectively cracked all identical ones, granting them access to multiple accounts at once.

A short salt, while less critical than a shared one, still increases the odds of different users having the same salt. This might create clusters
of password hashes with identical salt that can then be attacked as explained before.

With short salts, the probability of a collision between two users' passwords and salts couple might be low depending on the salt size. The shorter
the salt, the higher the collision probability. In any case, using longer, cryptographically secure salt should be preferred.

### Exceptions

To securely store password hashes, it is a recommended to rely on key derivation functions that are computationally intensive. Examples of such
functions are:

- Argon2
- PBKDF2
- Scrypt
- Bcrypt

When they are used for password storage, using a secure, random salt is required.

However, those functions can also be used for other purposes such as master key derivation or password-based pre-shared key generation. In those
cases, the implemented cryptographic protocol might require using a fixed salt to derive keys in a deterministic way. In such cases, using a fixed
salt is safe and accepted.

## How to fix it in .NET

### Code examples

The following code contains examples of hard-coded salts.

#### Noncompliant code example

    using System.Security.Cryptography;
    
    public static void hash(string password)
    {
        var salt = Encoding.UTF8.GetBytes("salty");
        var hashed = new Rfc2898DeriveBytes(password, salt); // Noncompliant
    }

#### Compliant solution

    using System.Security.Cryptography;
    
    public static void hash(string password)
    {
        var hashed = new Rfc2898DeriveBytes(password, 32, 10000, HashAlgorithmName.SHA256);
    }

### How does this work?

This code ensures that each user’s password has a unique salt value associated with it. It generates a salt randomly and with a length that
provides the required security level. It uses a salt length of at least 32 bytes (256 bits), as recommended by industry standards.

In the case of the code sample, the class automatically takes care of generating a secure salt if none is specified.

## Resources

### Standards

- OWASP - [Top 10 2021 Category A2 - Cryptographic Failures](https://owasp.org/Top10/A02_2021-Cryptographic_Failures/)
- OWASP - [Top 10 2017 Category A3 - Sensitive Data
  Exposure](https://www.owasp.org/www-project-top-ten/2017/A3_2017-Sensitive_Data_Exposure)
- CWE - [CWE-759 - Use of a One-Way Hash without a Salt](https://cwe.mitre.org/data/definitions/759)
- CWE - [CWE-760 - Use of a One-Way Hash with a Predictable Salt](https://cwe.mitre.org/data/definitions/760)
- STIG Viewer - [Application Security and
  Development: V-222542](https://stigviewer.com/stig/application_security_and_development/2023-06-08/finding/V-222542) - The application must only store cryptographic representations of passwords.