## Why is this an issue?

Casting expressions are utilized to convert one data type to another, such as transforming an integer into a string. This is especially crucial in
strongly typed languages like C, C++, C#, Java, Python, and others.

However, there are instances where casting expressions are not needed. These include situations like:

- casting a variable to its own type
- casting a subclass to a parent class (in the case of polymorphism)
- the programming language is capable of automatically converting the given type to another

These scenarios are considered unnecessary casting expressions. They can complicate the code and make it more difficult to understand, without
offering any advantages.

As a result, it’s generally advised to avoid unnecessary casting expressions. Instead, rely on the language’s type system to ensure type safety and
code clarity.

### Exceptions

Issues are not raised against the [default literal](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/types/casting-and-type-conversions).

## How to fix it

To fix your code remove the unnecessary casting expression.

### Code examples

#### Noncompliant code example

    public int Example(int i)
    {
        return (int) (i + 42); // Noncompliant
    }
    
    public IEnumerable<int> ExampleCollection(IEnumerable<int> coll)
    {
        return coll.Reverse().OfType<int>(); // Noncompliant
    }

#### Compliant solution

    public int Example(int i)
    {
        return i + 42;
    }
    
    public IEnumerable<int> ExampleCollection(IEnumerable<int> coll)
    {
        return coll.Reverse();
    }

    bool b = (bool)default; // Doesn't raise an issue

## Resources

### Documentation

- Microsoft - [Casting and type
  conversions](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/types/casting-and-type-conversions)
- Wikipedia - [Type Conversion](https://en.wikipedia.org/wiki/Type_conversion)
- Wikipedia - [Strong and Weak Typing](https://en.wikipedia.org/wiki/Strong_and_weak_typing)
- Wikipedia - [Polymorphism (Computer Science)](https://en.wikipedia.org/wiki/Polymorphism_%28computer_science%29)