## Why is this an issue?

A [loop statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/iteration-statements) with at most one
iteration is equivalent to an `if` statement; the following block is executed only once.

If the initial intention was to conditionally execute the block only once, an `if` statement should be used instead. If that was not the
initial intention, the block of the loop should be fixed so the block is executed multiple times.

A loop statement with at most one iteration can happen when a statement unconditionally transfers control, such as a [jump statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/jump-statements) or a [throw
statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements#the-throw-statement), is misplaced inside the loop block.

This rule raises when the following statements are misplaced:

- [`break`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/jump-statements#the-break-statement)
- [`continue`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/jump-statements#the-continue-statement)
- [`return`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/jump-statements#the-return-statement)
- [`throw`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements#the-throw-statement)

## How to fix it

### Code examples

#### Noncompliant code example

    public object Method(IEnumerable<object> items)
    {
        for (int i = 0; i < 10; i++)
        {
            Console.WriteLine(i);
            break; // Noncompliant: loop only executes once
        }
    
        foreach (object item in items)
        {
            return item; // Noncompliant: loop only executes once
        }
        return null;
    }

#### Compliant solution

    public object Method(IEnumerable<object> items)
    {
        for (int i = 0; i < 10; i++)
        {
            Console.WriteLine(i);
        }
    
        var item = items.FirstOrDefault();
        if (item != null)
        {
            return item;
        }
        return null;
    }

## Resources

### Documentation

- Microsoft Learn - [Iteration
  statements - `for`, `foreach`, `do`, and `while`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/iteration-statements)
- Microsoft Learn - [Jump statements -
  `break`, `continue`, `return`, and `goto`](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/jump-statements)
- Microsoft Learn - [The
  `throw` statement](https://learn.microsoft.com/en-us/dotnet/csharp/language-reference/statements/exception-handling-statements#the-throw-statement)