## Why is this an issue?

Calling an overridable method from a constructor could result in failures or strange behaviors when instantiating a subclass which overrides the
method.

When constructing an object of a derived class, the constructor of the parent class is invoked first, and only then the constructor of the derived
class is called. This sequential construction process applies to multiple levels of inheritance as well, starting from the base class and progressing
to the most derived class.

If an overridable method is called within the constructor of the parent class, it may inadvertently invoke an overridden implementation in the
derived class. This can lead to unexpected failures or strange behaviors because the object’s construction is still in progress and may not have
reached a fully initialized state. Consequently, the overridden method may rely on uninitialized members or have assumptions about the object’s state
that are not yet valid.

For example:

    public class Parent
    {
      public Parent()
      {
        DoSomething();  // Noncompliant
      }
    
      public virtual void DoSomething() // can be overridden
      {
        ...
      }
    }
    
    public class Child : Parent
    {
      private string foo;
    
      public Child(string foo) // leads to call DoSomething() in Parent constructor which triggers a NullReferenceException as foo has not yet been initialized
      {
        this.foo = foo;
      }
    
      public override void DoSomething()
      {
        Console.WriteLine(this.foo.Length);
      }
    }

- The `Child` class constructor starts by calling the `Parent` class constructor.
- The `Parent` class constructor calls the method `DoSomething`, which has been overridden in the `Child` class.
- If the behavior of the `Child` class overridden `DoSomething` method depends on fields that are initialized in the
  `Child` class constructor, unexpected behavior (such as a `NullReferenceException`) can result, because the fields aren’t
  initialized yet.

## Resources

### Documentation

- [Constructors](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/constructors)
- [Inheritance](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/object-oriented/inheritance)
- [Polimorphism](https://learn.microsoft.com/en-us/dotnet/csharp/fundamentals/object-oriented/polymorphism#virtual-methods)
- [Method signatures](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/methods#method-signatures)