## Why is this an issue?

Suppose you override [Object.Equals](https://learn.microsoft.com/en-us/dotnet/api/system.object.equals) in a type, you must also
override [Object.GetHashCode](https://learn.microsoft.com/en-us/dotnet/api/system.object.gethashcode). If two objects are equal according
to the `Equals` method, then calling `GetHashCode` on each of them must yield the same integer. If this is not the case, many
collections, such as a [Hashtable](https://learn.microsoft.com/en-us/dotnet/api/system.collections.hashtable) or a [Dictionary](https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.dictionary-2) won’t handle class instances correctly.

In order to not have unpredictable behavior, `Equals` and `GetHashCode` should be either both inherited, or both
overridden.

## How to fix it

When you override `Equals` then you have to also override `GetHashCode`. You have to override both of them, or simply inherit
them.

### Code examples

#### Noncompliant code example

    class MyClass   // Noncompliant: should also override GetHashCode
    {
        public override bool Equals(object obj)
        {
            // ...
        }
    }

#### Compliant solution

    class MyClass
    {
        public override bool Equals(object obj)
        {
            // ...
        }
    
        public override int GetHashCode()
        {
            // ...
        }
    }

## Resources

### Documentation

- CWE - [CWE-581 - Object Model Violation: Just One of Equals and Hashcode Defined](https://cwe.mitre.org/data/definitions/581)
- [Object.Equals Method](https://learn.microsoft.com/en-us/dotnet/api/system.object.equals)
- [Object.GetHashCode Method](https://learn.microsoft.com/en-us/dotnet/api/system.object.gethashcode)
- [Hashtable class](https://learn.microsoft.com/en-us/dotnet/api/system.collections.hashtable)
- [Dictionary&lt;TKey,TValue&gt; Class](https://learn.microsoft.com/en-us/dotnet/api/system.collections.generic.dictionary-2)