## Why is this an issue?

When you call `Any()`, it clearly communicates the code’s intention, which is to check if the collection is empty. Using `Count()
== 0` for this purpose is less direct and makes the code slightly more complex. However, there are some cases where special attention should be
paid:

- if the collection is an `EntityFramework` or other ORM query, calling `Count()` will cause executing a potentially
  massive SQL query and could put a large overhead on the application database. Calling `Any()` will also connect to the database, but will
  generate much more efficient SQL.
- if the collection is part of a LINQ query that contains `Select()` statements that create objects, a large amount of memory could be
  unnecessarily allocated. Calling `Any()` will be much more efficient because it will execute fewer iterations of the enumerable.

    private static bool HasContent(IEnumerable<string> strings)
    {
      return strings.Count() > 0;  // Noncompliant
    }
    
    private static bool HasContent2(IEnumerable<string> strings)
    {
      return strings.Count() >= 1;  // Noncompliant
    }
    
    private static bool IsEmpty(IEnumerable<string> strings)
    {
      return strings.Count() == 0;  // Noncompliant
    }

Prefer using `Any()` to test for emptiness over `Count()`.

    private static bool HasContent(IEnumerable<string> strings)
    {
      return strings.Any();
    }
    
    private static bool HasContent2(IEnumerable<string> strings)
    {
      return strings.Any();
    }
    
    private static bool IsEmpty(IEnumerable<string> strings)
    {
      return !strings.Any();
    }