## Why is this an issue?

Public fields in public classes do not respect the encapsulation principle and have three main disadvantages:

- Additional behavior such as validation cannot be added.
- The internal representation is exposed, and cannot be changed afterwards.
- Member values are subject to change from anywhere in the code and may not meet the programmer’s assumptions.

To prevent unauthorized modifications, private attributes and accessor methods (set and get) should be used.

Note that due to optimizations on simple properties, public fields provide only very little performance gain.

### What is the potential impact?

Public fields can be modified by any part of the code and this can lead to unexpected changes and hard-to-trace bugs.

Public fields don’t hide the implementation details. As a consequence, it is no longer possible to change how the data is stored internally without
impacting the client code of the class.

The code is harder to maintain.

### Exceptions

Fields marked as `readonly` or `const` are ignored by this rule.

Fields inside classes or structs annotated with `[StructLayout]` are ignored by this rule.

Fields inside classes or structs annotated with `[Serializable]` are ignored by this rule unless they are annotated with
`[NonSerialized]`.

## How to fix it

Depending on your needs:

- Use auto-implemented properties:  
 For common cases, where no validation is required, auto-implemented properties are a good alternative to
  fields: these allows fine grained access control and offers the flexibility to add validation or change internal storage afterwards. *Note:*
  as a bonus it is now possible to monitor value changes using breakpoints.
- Encapsulate the fields in your class. To do so:
    1. Make the field private.
    2. Use public properties (set and get) to access and modify the field.
- Mark field as `readonly` or `const`.

### Code examples

#### Noncompliant code example

    public class Foo
    {
        public int InstanceData = 32; // Noncompliant
        public int AnotherInstanceData = 32; // Noncompliant
    
    }

#### Compliant solution

    public class Foo
    {
        // using auto-implemented properties
        public int InstanceData { get; set; } = 32;
    
        // using field encapsulation
        private int _anotherInstanceData = 32;
    
        public int AnotherInstanceData
        {
            get { return _anotherInstanceData; }
            set
            {
                // perform validation
                _anotherInstanceData = value;
            }
        }
    
    }

### Pitfalls

Please be aware that changing a field by a property in a software that uses serialization could lead to binary incompatibility.

## Resources

- CWE - [CWE-493 - Critical Public Variable Without Final Modifier](https://cwe.mitre.org/data/definitions/493)