## Why is this an issue?

The [finalizers](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/finalizers) are used to perform
[any necessary final clean-up](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/fundamentals#unmanaged-resources) when
the garbage collector is collecting a class instance. The programmer has no control over when the finalizer is called; the garbage collector decides
when to call it.

When creating a finalizer, it should never throw an exception, as there is a high risk of having the application terminated leaving unmanaged
resources without a graceful cleanup.

The rule raises an issue on `throw` statements used in a finalizer.

## How to fix it

### Code examples

#### Noncompliant code example

    class MyClass
    {
        ~MyClass()
        {
            throw new NotImplementedException(); // Noncompliant: finalizer throws an exception
        }
    }

#### Compliant solution

    class MyClass
    {
        ~MyClass()
        {
            // Compliant: finalizer does not throw an exception
        }
    }

### Going the extra mile

In general object finalization can be a complex and error-prone operation and should not be implemented except within the [dispose pattern](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/implementing-dispose).

When [cleaning up unmanaged resources](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/unmanaged), it is
recommended to implement the dispose pattern or, to cover uncalled [`Dispose`](https://learn.microsoft.com/en-us/dotnet/api/system.idisposable.dispose) method by the consumer, implement [`SafeHandle`](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.safehandle).

## Resources

### Documentation

- Microsoft Learn - [Fundamentals of garbage
  collection](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/fundamentals)
- Microsoft Learn - [Cleaning up unmanaged resources](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/unmanaged)
- Microsoft Learn - [Implement a Dispose
  method](https://learn.microsoft.com/en-us/dotnet/standard/garbage-collection/implementing-dispose)
- Microsoft Learn - [`SafeHandle`
  Class](https://learn.microsoft.com/en-us/dotnet/api/system.runtime.interopservices.safehandle)
- Microsoft Learn - [`IDisposable.Dispose` Method](https://learn.microsoft.com/en-us/dotnet/api/system.idisposable.dispose)
- Microsoft Learn - [Finalizers
  (destructors)](https://learn.microsoft.com/en-us/dotnet/csharp/programming-guide/classes-and-structs/finalizers)