/* Copyright (c) 2019-2020 Cameron Harper
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
 * COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * */

#include "ldl_stream.h"
#include "ldl_debug.h"
#include "ldl_internal.h"

#include <string.h>

void LDL_Stream_init(struct ldl_stream *self, void *buf, uint8_t size)
{
    LDL_PEDANTIC(self != NULL)
    LDL_PEDANTIC((buf != NULL) || (size == 0U))

    self->write = buf;
    self->read = buf;
    self->size = size;
    self->pos = 0U;
    self->error = false;
}

void LDL_Stream_initReadOnly(struct ldl_stream *self, const void *buf, uint8_t size)
{
    LDL_PEDANTIC(self != NULL)
    LDL_PEDANTIC((buf != NULL) || (size == 0U))

    self->write = NULL;
    self->read = buf;
    self->size = size;
    self->pos = 0U;
    self->error = false;
}

bool LDL_Stream_read(struct ldl_stream *self, void *buf, uint8_t count)
{
    LDL_PEDANTIC(self != NULL)
    LDL_PEDANTIC(buf != NULL)

    bool retval = false;

    if(!self->error){

        if((self->size - self->pos) >= count){

            (void)memcpy(buf, &self->read[self->pos], count);
            self->pos += count;
            retval = true;
        }
        else{

            self->error = true;
        }
    }

    return retval;
}

bool LDL_Stream_write(struct ldl_stream *self, const void *buf, uint8_t count)
{
    LDL_PEDANTIC(self != NULL)

    bool retval = false;

    if(self->write != NULL){

        if(!self->error){

            if((self->size - self->pos) >= count){

                (void)memcpy(&self->write[self->pos], buf, count);
                self->pos += count;
                retval = true;
            }
            else{

                self->error = true;
            }
        }
    }

    return retval;
}

uint8_t LDL_Stream_tell(const struct ldl_stream *self)
{
    LDL_PEDANTIC(self != NULL)

    return self->pos;
}

uint8_t LDL_Stream_remaining(const struct ldl_stream *self)
{
    LDL_PEDANTIC(self != NULL)

    return (self->size - self->pos);
}

bool LDL_Stream_peek(const struct ldl_stream *self, void *out)
{
    LDL_PEDANTIC(self != NULL)

    bool retval = false;

    if(LDL_Stream_remaining(self) > 0U){

        (void)memcpy(out, &self->read[self->pos], 1U);
        retval = true;
    }

    return retval;
}

bool LDL_Stream_seekSet(struct ldl_stream *self, uint8_t offset)
{
    LDL_PEDANTIC(self != NULL)

    bool retval = false;

    if(self->size >= offset){

        self->pos = offset;
        retval = true;
    }

    return retval;
}

void LDL_Stream_rewind(struct ldl_stream *self)
{
    LDL_PEDANTIC(self != NULL)

    self->pos = 0;
}

bool LDL_Stream_seekCur(struct ldl_stream *self, int16_t offset)
{
    LDL_PEDANTIC(self != NULL)

    bool retval = false;

    int32_t pos = ((int32_t)self->pos) + ((int32_t)offset);
    int32_t size = (int32_t)self->size;

    if((pos >= 0) && (pos <= size)){

        if(offset >= 0){

            self->pos += U8(offset);
        }
        else{

            self->pos -= U8(offset);
        }

        retval = true;
    }

    return retval;
}

bool LDL_Stream_putU8(struct ldl_stream *self, uint8_t value)
{
    LDL_PEDANTIC(self != NULL)

    return LDL_Stream_write(self, &value, 1U);
}

bool LDL_Stream_putU16(struct ldl_stream *self, uint16_t value)
{
    LDL_PEDANTIC(self != NULL)

    bool retval;

#ifdef LDL_LITTLE_ENDIAN
    retval = LDL_Stream_write(self, &value, 2U);
#else
    uint8_t out[] = {
        U8(value),
        U8(value >> 8)
    };

    retval = LDL_Stream_write(self, out, U8(sizeof(out)));
#endif

    return retval;
}

bool LDL_Stream_putU24(struct ldl_stream *self, uint32_t value)
{
    LDL_PEDANTIC(self != NULL)

    bool retval;

#ifdef LDL_LITTLE_ENDIAN
    retval = LDL_Stream_write(self, &value, 3U);
#else
    uint8_t out[] = {
        U8(value),
        U8(value >> 8),
        U8(value >> 16)
    };

    retval = LDL_Stream_write(self, out, U8(sizeof(out)));
#endif

    return retval;
}

bool LDL_Stream_putU32(struct ldl_stream *self, uint32_t value)
{
    LDL_PEDANTIC(self != NULL)

    bool retval;

#ifdef LDL_LITTLE_ENDIAN
    retval = LDL_Stream_write(self, &value, 4U);
#else
    uint8_t out[] = {
        U8(value),
        U8(value >> 8),
        U8(value >> 16),
        U8(value >> 24)
    };

    retval = LDL_Stream_write(self, out, U8(sizeof(out)));
#endif

    return retval;
}

bool LDL_Stream_putEUI(struct ldl_stream *self, const uint8_t *value)
{
    LDL_PEDANTIC(self != NULL)

    uint8_t out[] = {
        value[7],
        value[6],
        value[5],
        value[4],
        value[3],
        value[2],
        value[1],
        value[0],
    };

    return LDL_Stream_write(self, out, U8(sizeof(out)));
}

bool LDL_Stream_getU8(struct ldl_stream *self, uint8_t *value)
{
    LDL_PEDANTIC(self != NULL)

    return LDL_Stream_read(self, value, U8(sizeof(*value)));
}

bool LDL_Stream_getU16(struct ldl_stream *self, uint16_t *value)
{
    LDL_PEDANTIC(self != NULL)

    bool retval;

#ifdef LDL_LITTLE_ENDIAN
    retval = LDL_Stream_read(self, value, 2U);
#else
    uint8_t buf[2];

    retval = LDL_Stream_read(self, buf, U8(sizeof(buf)));

    if(retval){

        *value = buf[1];
        *value <<= 8;
        *value |= buf[0];
    }
#endif

    return retval;
}

bool LDL_Stream_getU24(struct ldl_stream *self, uint32_t *value)
{
    LDL_PEDANTIC(self != NULL)

    bool retval;

#ifdef LDL_LITTLE_ENDIAN
    *value = 0U;
    retval = LDL_Stream_read(self, value, 3U);
#else
    uint8_t buf[3];

    retval = LDL_Stream_read(self, buf, U8(sizeof(buf)));

    if(retval){

        *value = buf[2];
        *value <<= 8;
        *value |= buf[1];
        *value <<= 8;
        *value |= buf[0];
    }
#endif
    return retval;
}

bool LDL_Stream_getU32(struct ldl_stream *self, uint32_t *value)
{
    LDL_PEDANTIC(self != NULL)

    bool retval;

#ifdef LDL_LITTLE_ENDIAN
    retval = LDL_Stream_read(self, value, 4U);
#else
    uint8_t buf[4];

    retval = LDL_Stream_read(self, buf, U8(sizeof(buf)));

    if(retval){

        *value = buf[3];
        *value <<= 8;
        *value |= buf[2];
        *value <<= 8;
        *value |= buf[1];
        *value <<= 8;
        *value |= buf[0];
    }
#endif

    return retval;
}

bool LDL_Stream_getEUI(struct ldl_stream *self, uint8_t *value)
{
    LDL_PEDANTIC(self != NULL)

    uint8_t out[8];
    bool retval;

    retval = LDL_Stream_read(self, out, U8(sizeof(out)));

    if(retval){

        value[0] = out[7];
        value[1] = out[6];
        value[2] = out[5];
        value[3] = out[4];
        value[4] = out[3];
        value[5] = out[2];
        value[6] = out[1];
        value[7] = out[0];
    }

    return retval;
}

bool LDL_Stream_error(struct ldl_stream *self)
{
    LDL_PEDANTIC(self != NULL)

    return self->error;
}
