package io.cequence.openaiscala.domain

sealed trait BaseMessage {
  val role: ChatRole
  val nameOpt: Option[String]
  def isSystem: Boolean = role == ChatRole.System
}

final case class SystemMessage(
  // The contents of the message.
  content: String,

  // An optional name for the participant. Provides the model information to differentiate between participants of the same role.
  // May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
  name: Option[String] = None
) extends BaseMessage {
  override val role = ChatRole.System
  override val nameOpt = name

  def withName(name: String): SystemMessage = this.copy(name = Some(name))
}

final case class UserMessage(
  // The contents of the message.
  content: String,

  // An optional name for the participant. Provides the model information to differentiate between participants of the same role.
  // May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
  name: Option[String] = None
) extends BaseMessage {
  override val role = ChatRole.User
  override val nameOpt = name

  def withName(name: String): UserMessage = this.copy(name = Some(name))
}

final case class UserSeqMessage(
  // The contents of the message - text or image URL.
  content: Seq[Content],

  // An optional name for the participant. Provides the model information to differentiate between participants of the same role.
  // May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
  name: Option[String] = None
) extends BaseMessage {
  override val role = ChatRole.User
  override val nameOpt = name
}

final case class AssistantMessage(
  // The contents of the message.
  content: String,

  // An optional name for the participant. Provides the model information to differentiate between participants of the same role.
  // May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
  name: Option[String] = None
) extends BaseMessage {
  override val role = ChatRole.Assistant
  override val nameOpt = name

  def withName(name: String): AssistantMessage = this.copy(name = Some(name))
}

final case class AssistantToolMessage(
  // The contents of the message.
  content: Option[String] = None,

  // An optional name for the participant. Provides the model information to differentiate between participants of the same role.
  // May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
  name: Option[String] = None,

  // The tool calls generated by the model, such as function calls defined as id-call pairs
  tool_calls: Seq[(String, ToolCallSpec)] = Nil
) extends BaseMessage {
  override val role = ChatRole.Assistant
  override val nameOpt = name
}

final case class AssistantToolOutput(
  tool_call_id: String,
  // The contents of the message.
  output: Option[String] = None
)

@Deprecated
final case class AssistantFunMessage(
  // The contents of the message.
  content: Option[String] = None,

  // An optional name for the participant. Provides the model information to differentiate between participants of the same role.
  // May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
  name: Option[String] = None,

  // Deprecated and replaced by tool_calls. The name and arguments of a function that should be called, as generated by the model.
  function_call: Option[FunctionCallSpec] = None
) extends BaseMessage {
  override val role = ChatRole.Assistant
  override val nameOpt = name
}

final case class ToolMessage(
  // Tool/function response
  content: Option[String] = None, // Is it mandatory?

  // Tool call that this message is responding to.
  tool_call_id: String,

  // Tool/function name
  name: String
) extends BaseMessage {
  override val role = ChatRole.Tool
  override val nameOpt = Some(name)
}

@Deprecated
final case class FunMessage(
  // Function response
  content: String,

  // Function name
  name: String
) extends BaseMessage {
  override val role = ChatRole.Function
  override val nameOpt = Some(name)
}

/**
 * Deprecation warning: Use typed Message(s), such as SystemMessage, UserMessage, instead. Will
 * be dropped in the next major version.
 */
@Deprecated
final case class MessageSpec(
  // The role of the messages author. One of system, user, or assistant.
  role: ChatRole,

  // The contents of the message.
  content: String,

  // The name of the author of this message. name is required if role is function, and
  // it should be the name of the function whose response is in the content.
  // May contain a-z, A-Z, 0-9, and underscores, with a maximum length of 64 characters.
  name: Option[String] = None
) extends BaseMessage {
  override val nameOpt = name
}
