use mago_ast::ast::*;
use mago_reporting::*;
use mago_span::*;
use mago_walker::Walker;

use crate::context::LintContext;
use crate::rule::Rule;

#[derive(Clone, Copy, Debug)]
pub struct InterfaceRule;

impl Rule for InterfaceRule {
    fn get_name(&self) -> &'static str {
        "interface"
    }

    fn get_default_level(&self) -> Option<Level> {
        Some(Level::Help)
    }
}

impl<'a> Walker<LintContext<'a>> for InterfaceRule {
    fn walk_in_interface<'ast>(&self, interface: &'ast Interface, context: &mut LintContext<'a>) {
        let mut issues = vec![];

        let name = context.lookup(&interface.name.value);
        let fqcn = context.lookup_name(&interface.name);

        if !mago_casing::is_class_case(name) {
            issues.push(
                Issue::new(context.level(), format!("Interface name `{}` should be in class case.", name))
                    .with_annotations([
                        Annotation::primary(interface.name.span())
                            .with_message(format!("Interface `{}` is declared here.", name)),
                        Annotation::secondary(interface.span())
                            .with_message(format!("Interface `{}` is defined here.", fqcn)),
                    ])
                    .with_note(format!("The interface name `{}` does not follow class naming convention.", name))
                    .with_help(format!(
                        "Consider renaming it to `{}` to adhere to the naming convention.",
                        mago_casing::to_class_case(name)
                    )),
            );
        }

        if context.option("psr").and_then(|o| o.as_bool()).unwrap_or(true) && !name.ends_with("Interface") {
            issues.push(
                Issue::new(context.level(), format!("interface name `{}` should be suffixed with `Interface`.", name))
                    .with_annotations([
                        Annotation::primary(interface.name.span())
                            .with_message(format!("Interface `{}` is declared here.", name)),
                        Annotation::secondary(interface.span())
                            .with_message(format!("Interface `{}` is defined here.", fqcn)),
                    ])
                    .with_note(format!("The interface name `{}` does not follow PSR naming convention.", name))
                    .with_help(format!(
                        "Consider renaming it to `{}Interface` to adhere to the naming convention.",
                        name
                    )),
            );
        }

        for issue in issues {
            context.report(issue);
        }
    }
}
