import { type IntegrationDefinition, type BotDefinition, type InterfaceDeclaration } from '@botpress/sdk'
import { prepareCreateBotBody } from '../api/bot-body'
import { prepareCreateIntegrationBody } from '../api/integration-body'
import { prepareCreateInterfaceBody } from '../api/interface-body'
import type commandDefinitions from '../command-definitions'
import * as errors from '../errors'
import { BotLinter } from '../linter/bot-linter'
import { IntegrationLinter } from '../linter/integration-linter'
import { InterfaceLinter } from '../linter/interface-linter'
import { getImplementationStatements } from '../sdk'
import { ProjectCommand } from './project-command'

export type LintCommandDefinition = typeof commandDefinitions.lint
export class LintCommand extends ProjectCommand<LintCommandDefinition> {
  public async run(): Promise<void> {
    const projectDef = await this.readProjectDefinitionFromFS()
    if (projectDef.bpLintDisabled) {
      this.logger.warn(
        'Linting is disabled for this project because of a bplint directive. To enable linting, remove the "bplint-disable" directive from the project definition file'
      )
      return
    }

    switch (projectDef.type) {
      case 'integration':
        return this._runLintForIntegration(projectDef.definition)
      case 'bot':
        return this._runLintForBot(projectDef.definition)
      case 'interface':
        return this._runLintForInterface(projectDef.definition)
      default:
        throw new errors.BotpressCLIError('Unsupported project type')
    }
  }

  private async _runLintForInterface(definition: InterfaceDeclaration): Promise<void> {
    const parsedInterfaceDefinition = await prepareCreateInterfaceBody(definition)
    const linter = new InterfaceLinter(parsedInterfaceDefinition)

    await linter.lint()
    linter.logResults(this.logger)

    if (linter.hasErrors()) {
      throw new errors.BotpressCLIError('Interface definition contains linting errors')
    }

    this.logger.success('Interface definition is valid')
  }

  private async _runLintForBot(definition: BotDefinition): Promise<void> {
    const strippedDefinition = this._stripAutoGeneratedContentFromBot(definition)
    const parsedBotDefinition = await prepareCreateBotBody(strippedDefinition)
    const linter = new BotLinter(parsedBotDefinition)

    await linter.lint()
    linter.logResults(this.logger)

    if (linter.hasErrors()) {
      throw new errors.BotpressCLIError('Bot definition contains linting errors')
    }

    this.logger.success('Bot definition is valid')
  }

  private _stripAutoGeneratedContentFromBot(definition: BotDefinition) {
    return {
      ...definition,
      integrations: {},
    } as BotDefinition
  }

  private async _runLintForIntegration(definition: IntegrationDefinition): Promise<void> {
    const strippedDefinition = this._stripAutoGeneratedContentFromIntegration(definition)
    const parsedIntegrationDefinition = await prepareCreateIntegrationBody(strippedDefinition)
    const linter = new IntegrationLinter({ ...parsedIntegrationDefinition, secrets: strippedDefinition.secrets })

    await linter.lint()
    linter.logResults(this.logger)

    if (linter.hasErrors()) {
      throw new errors.BotpressCLIError('Integration definition contains linting errors')
    }

    this.logger.success('Integration definition is valid')
  }

  private _stripAutoGeneratedContentFromIntegration(definition: IntegrationDefinition) {
    const { actionNames, eventNames } = this._getAutoGeneratedContentOfIntegration(definition)

    return {
      ...definition,
      actions: Object.fromEntries(Object.entries(definition.actions ?? {}).filter(([key]) => !actionNames.has(key))),
      events: Object.fromEntries(Object.entries(definition.events ?? {}).filter(([key]) => !eventNames.has(key))),
    } as IntegrationDefinition
  }

  private _getAutoGeneratedContentOfIntegration(definition: IntegrationDefinition) {
    const actionNames = new Set<string>()
    const eventNames = new Set<string>()

    const interfacesStatements = getImplementationStatements(definition)
    for (const iface of Object.values(interfacesStatements)) {
      for (const actionDefinition of Object.values(iface.actions)) {
        actionNames.add(actionDefinition.name)
      }
      for (const eventDefinition of Object.values(iface.events)) {
        eventNames.add(eventDefinition.name)
      }
    }

    return { actionNames, eventNames } as const
  }
}
