import * as client from '@botpress/client'
import * as sdk from '@botpress/sdk'
import _ from 'lodash'

// TODO: these types should be generated by the CLI from interface and integration schemas

type BaseItem = { id: string }

export type DataSourceEvents<T extends BaseItem> = {
  created: { item: T }
  updated: { item: T }
  deleted: { item: T }
}

export type EventInput<T extends BaseItem> = {
  [K in keyof DataSourceEvents<T>]: {
    event: DataSourceEvents<T>[K]
    state: SyncerState
  }
}

export type DataSource<T extends BaseItem> = {
  list: (input: { nextToken?: string }) => Promise<{ items: T[]; meta: { nextToken?: string } }>
  create?: (input: { item: T }) => Promise<{ item: T }> // only needed for the 2-way sync
  update?: (input: { item: T }) => Promise<{ item: T }> // only needed for the 2-way sync
  delete?: (input: { id: string }) => Promise<{ id: string }> // only needed for the 2-way sync
  on: <K extends keyof DataSourceEvents<T>>(event: K, handler: (args: EventInput<T>[K]) => Promise<void>) => void
}

export type SyncerOptions = {
  tableName: string
}

export type SyncerState = {
  nextToken?: string
  tableCreated: boolean
}

const TABLE_RESERVED_KEYWORDS = ['id', 'createdAt', 'updatedAt']

export class Syncer<T extends BaseItem> {
  public constructor(
    private _dataSource: DataSource<T>,
    private _client: client.Client,
    private _options: SyncerOptions
  ) {
    _dataSource.on('created', this._onCreated)
    _dataSource.on('updated', this._onUpdated)
    _dataSource.on('deleted', this._onDeleted)
  }

  public async sync(state: SyncerState): Promise<SyncerState> {
    const { items, meta } = await this._dataSource.list(state)
    state = await this._upsert(state, items)
    return { ...state, nextToken: meta.nextToken }
  }

  private _onCreated = async (args: EventInput<T>['created']) => {
    await this._upsert(args.state, [args.event.item])
  }

  private _onUpdated = async (args: EventInput<T>['updated']) => {
    await this._upsert(args.state, [args.event.item])
  }

  private _onDeleted = async (args: EventInput<T>['deleted']) => {
    await this._delete(args.state, args.event.item.id)
  }

  private _upsert = async (state: SyncerState, items: T[]): Promise<SyncerState> => {
    if (!items.length) {
      return state
    }

    if (!state?.tableCreated) {
      const model = this._mergeObjects(items)
      await this._createTableIfNotExists(model)
      return { ...state, tableCreated: true }
    }

    const rows = items.map(this._escapeObject)
    const { errors } = await this._client.upsertTableRows({
      table: this._options.tableName,
      rows,
      keyColumn: '_id',
    })

    if (errors?.length) {
      const message = errors.join('; ')
      throw new sdk.RuntimeError(`Failed to upsert rows: ${message}`)
    }

    return state
  }

  private _delete = async (state: SyncerState, id: string): Promise<SyncerState> => {
    if (!state?.tableCreated) {
      return state ?? {}
    }

    await this._client.deleteTableRows({
      table: this._options.tableName,
      filter: { id: { $eq: id } },
    })

    return state
  }

  private _createTableIfNotExists = async (model: object): Promise<void> => {
    const escapedModel = this._escapeObject(model)
    const { tables } = await this._client.listTables({})
    const existingTable = tables.find((table) => table.name === this._options.tableName)
    if (!existingTable) {
      await this._client.createTable({
        name: this._options.tableName,
        schema: escapedModel, // should pass a json schema here, but the code generation of the CLI doesn't support it yet
      })
    }
  }

  private _mergeObjects = (objs: object[]): object => {
    return objs.reduce(
      (acc, obj) => _.mergeWith(acc, obj, (accVal, objVal) => (accVal === undefined ? objVal : accVal)),
      {}
    )
  }

  private _escapeObject = (obj: object): object => {
    return _(obj)
      .toPairs()
      .map(([key, value]) => [this._escapeKey(key), value])
      .fromPairs()
      .value()
  }

  private _unescapeObject = (obj: object): object => {
    return _(obj)
      .toPairs()
      .map(([key, value]) => [this._unescapeKey(key), value])
      .fromPairs()
      .value()
  }

  private _escapeKey = (key: string): string => (TABLE_RESERVED_KEYWORDS.includes(key) ? `_${key}` : key)
  private _unescapeKey = (key: string): string => {
    const escapedColumns = TABLE_RESERVED_KEYWORDS.map(this._escapeKey)
    return escapedColumns.includes(key) ? key.slice(1) : key
  }
}
