import os
import os.path

from .lib.config import get_config
from .lib.translation import get_engine_class
from .lib.conversion import extra_formats
from .lib.encodings import encoding_list
from .engines.custom import CustomTranslate
from .components import (
    Footer, AlertMessage, SourceLang, TargetLang, InputFormat,
    OutputFormat)


try:
    from qt.core import (
        QDialog, QWidget, QPushButton, QHeaderView, QVBoxLayout, QTableWidget,
        QTableWidgetItem, Qt, QComboBox, QLabel, QSizePolicy, QHBoxLayout)
except ImportError:
    from PyQt5.Qt import (
        QDialog, QWidget, QPushButton, QHeaderView, QVBoxLayout, QTableWidget,
        QTableWidgetItem, Qt, QComboBox, QLabel, QSizePolicy, QHBoxLayout)

load_translations()


class BatchTranslation(QDialog):
    def __init__(self, parent, worker, ebooks):
        QDialog.__init__(self, parent)

        self.gui = parent
        self.worker = worker
        self.ebooks = ebooks
        self.alert = AlertMessage(self)

        self.config = get_config()
        self.jobs = {}
        self.source_langs = []
        self.target_langs = []

        self.main_layout()

    def main_layout(self):
        layout = QVBoxLayout(self)
        layout.addWidget(self.layout_translate())
        layout.addWidget(Footer())

    def _cell_widget(self, _widget):
        widget = QWidget()
        layout = QHBoxLayout(widget)
        layout.setContentsMargins(5, 5, 5, 5)
        # _widget.setFixedSize(_widget.sizeHint())
        # _widget.setFixedHeight(_widget.sizeHint().height())
        layout.addWidget(_widget, 1)
        # layout.setAlignment(_widget, Qt.AlignCenter)
        return widget

    def layout_translate(self):
        widget = QWidget()
        layout = QVBoxLayout(widget)
        layout.setContentsMargins(0, 0, 0, 0)

        table = QTableWidget()
        table.setAlternatingRowColors(True)
        table.setFocusPolicy(Qt.NoFocus)
        # table.setSelectionMode(QTableWidget.NoSelection)
        table.setEditTriggers(QTableWidget.NoEditTriggers)
        table.setSelectionBehavior(QTableWidget.SelectRows)
        table.setRowCount(len(self.ebooks))

        headers = (
            _('Title'), _('Encoding'), _('Input Format'), _('Output Format'),
            _('Source Language'), _('Target Language'),
            _('Target Directionality'))
        table.setColumnCount(len(headers))
        table.setHorizontalHeaderLabels(headers)

        header = table.horizontalHeader()
        stretch = getattr(QHeaderView.ResizeMode, 'Stretch', None) \
            or QHeaderView.Stretch
        header.setSectionResizeMode(0, stretch)
        table.verticalHeader().setMaximumSectionSize(36)

        translation_engine = get_engine_class()
        for row, ebook in enumerate(self.ebooks):
            ebook_title = QTableWidgetItem(ebook.title)
            table.setItem(row, 0, ebook_title)

            if ebook.input_format in extra_formats.keys():
                input_encoding = QComboBox()
                input_encoding.wheelEvent = lambda event: None
                table.setCellWidget(row, 1, self._cell_widget(input_encoding))
                input_encoding.addItems(encoding_list)
                input_encoding.currentTextChanged.connect(
                    lambda encoding, row=row: self.ebooks[row]
                    .set_encoding(encoding))
                input_encoding.currentTextChanged.connect(
                    lambda encoding: input_encoding.setToolTip(encoding))
                # Target directionality
                target_direction = QTableWidgetItem(_('Default'))
                target_direction.setTextAlignment(Qt.AlignCenter)
                table.setItem(row, 6, target_direction)
            else:
                input_encoding = QTableWidgetItem(_('Default'))
                input_encoding.setTextAlignment(Qt.AlignCenter)
                table.setItem(row, 1, input_encoding)
                # Target directionality
                direction_list = QComboBox()
                direction_list.wheelEvent = lambda event: None
                direction_list.addItem(_('Auto'), 'auto')
                direction_list.addItem(_('Left to Right'), 'ltr')
                direction_list.addItem(_('Right to Left'), 'rtl')
                direction_list.currentIndexChanged.connect(
                    lambda index, row=row: self.ebooks[row]
                    .set_target_direction(direction_list.itemData(index)))
                direction_list.currentTextChanged.connect(
                    lambda direction: direction_list.setToolTip(direction))
                table.setCellWidget(row, 6, self._cell_widget(direction_list))

            input_fmt = InputFormat(ebook.files.keys())
            table.setCellWidget(row, 2, self._cell_widget(input_fmt))

            output_format = OutputFormat()
            table.setCellWidget(row, 3, self._cell_widget(output_format))

            existed_format = output_format.findText(ebook.input_format)
            if ebook.is_extra_format() and existed_format:
                output_format.addItem(ebook.input_format)

            def change_input_format(format, row=row):
                ebook = self.ebooks[row]
                ebook.set_input_format(format)
                if ebook.is_extra_format():
                    output_format.lock_format(format)
                else:
                    output_format.unlock_format()
                input_fmt.setToolTip(format)
            change_input_format(input_fmt.currentText(), row)
            input_fmt.currentTextChanged.connect(change_input_format)

            def change_output_format(format, row=row):
                self.ebooks[row].set_output_format(format)
                output_format.setToolTip(format)
            change_output_format(output_format.currentText(), row)
            output_format.currentTextChanged.connect(change_output_format)

            source_lang = SourceLang(book_lang=ebook.source_lang)
            table.setCellWidget(row, 4, self._cell_widget(source_lang))
            self.source_langs.append(source_lang)

            def change_source_lang(lang, row=row):
                self.ebooks[row].set_source_lang(lang)
                source_lang.setToolTip(lang)
            change_source_lang(source_lang.currentText(), row)
            source_lang.currentTextChanged.connect(change_source_lang)

            source_lang.refresh.emit(
                translation_engine.lang_codes.get('source'),
                translation_engine.config.get('source_lang'),
                not issubclass(translation_engine, CustomTranslate))

            target_lang = TargetLang()
            table.setCellWidget(row, 5, self._cell_widget(target_lang))
            self.target_langs.append(target_lang)

            def change_target_lang(lang, row=row):
                ebook = self.ebooks[row]
                ebook.set_target_lang(lang)
                ebook.set_lang_code(
                    translation_engine.get_iso639_target_code(lang))
                target_lang.setToolTip(lang)
            change_target_lang(target_lang.currentText(), row)
            target_lang.currentTextChanged.connect(change_target_lang)

            target_lang.refresh.emit(
                translation_engine.lang_codes.get('target'),
                translation_engine.config.get('target_lang'))

            table.resizeRowsToContents()
            table.resizeColumnsToContents()

        layout.addWidget(table)

        start_button = QPushButton(_('Translate'))
        start_button.setStyleSheet(
            'padding:0;height:48;font-size:20px;color:royalblue;'
            'text-transform:uppercase;')
        start_button.clicked.connect(
            lambda: self.translate_ebooks(self.ebooks))
        layout.addWidget(start_button)

        # Change the book title
        table.itemChanged.connect(
            lambda item: self.alter_ebooks_data(
                item.row(), item.text().strip() or _('Unknown')))

        return widget

    def translate_ebooks(self, ebooks):
        to_library = self.config.get('to_library')
        output_path = self.config.get('output_path')
        if not to_library and not os.path.exists(output_path):
            return self.alert.pop(
                _('The specified path does not exist.'), 'warning')
        ebooks = ebooks if isinstance(ebooks, list) else [ebooks]
        for ebook in self.ebooks:
            self.worker.translate_ebook(ebook, is_batch=True)
        self.ebooks.clear()
        self.done(0)
