/*
 * Copyright (C) 2023 Amazon.com, Inc. or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import * as iam from 'aws-cdk-lib/aws-iam';
import * as sqs from 'aws-cdk-lib/aws-sqs';

import { Construct } from 'constructs';
import { ServiceDescription } from '@project-lakechain/core/service';
import { ComputeType } from '@project-lakechain/core/compute-type';
import { SqsStorageConnectorProps, SqsStorageConnectorPropsSchema } from './definitions/opts';
import {
  Middleware,
  MiddlewareBuilder
} from '@project-lakechain/core/middleware';

/**
 * The service description.
 */
const description: ServiceDescription = {
  name: 'sqs-storage-connector',
  description: 'Stores documents and their metadata in an SQS queue.',
  version: '0.10.0',
  attrs: {}
};

/**
 * Builder for the `SqsStorageConnector` middleware.
 */
class SqsStorageConnectorBuilder extends MiddlewareBuilder {
  private providerProps: Partial<SqsStorageConnectorProps> = {};

  /**
   * Specifies the destination SQS queue for the
   * processed documents.
   * @param destination the destination queue.
   */
  public withDestinationQueue(destination: sqs.Queue) {
    this.providerProps.destinationQueue = destination;
    return (this);
  }

  /**
   * @returns a new instance of the `SqsStorageConnector`
   * service constructed with the given parameters.
   */
  public build(): SqsStorageConnector {
    return (new SqsStorageConnector(
      this.scope,
      this.identifier, {
        ...this.providerProps as SqsStorageConnectorProps,
        ...this.props
      }
    ));
  }
}

/**
 * A middleware allowing to forward document events
 * to an SQS queue.
 */
export class SqsStorageConnector extends Middleware {

  /**
   * The builder for the `SqsStorageConnector` service.
   */
  public static readonly Builder = SqsStorageConnectorBuilder;

  /**
   * Provider constructor.
   */
  constructor(scope: Construct, id: string, private props: SqsStorageConnectorProps) {
    super(scope, id, description, props);

    // Validating the properties.
    this.props = this.parse(SqsStorageConnectorPropsSchema, props);

    super.bind();
  }

  /**
   * We override the `getInput` method to return the user-provided
   * queue. This way this middleware only acts as a passthrough between
   * the previous middleware's SNS topics and the user-provided queue.
   * @returns the user-provided SQS queue.
   */
  public getInput() {
    return (this.props.destinationQueue);
  }

  /**
   * Allows a grantee to read from the processed documents
   * generated by this middleware.
   */
  grantReadProcessedDocuments(_: iam.IGrantable): iam.Grant {
    return ({} as iam.Grant);
  }

  /**
   * @returns an array of mime-types supported as input
   * type by the data producer.
   */
  supportedInputTypes(): string[] {
    return ([
      '*/*'
    ]);
  }

  /**
   * @returns an array of mime-types supported as output
   * type by the data producer.
   */
  supportedOutputTypes(): string[] {
    return ([]);
  }

  /**
   * @returns the supported compute types by a given
   * middleware.
   */
  supportedComputeTypes(): ComputeType[] {
    return ([
      ComputeType.CPU
    ]);
  }
}
