/*
 * Copyright (C) 2023 Amazon.com, Inc. or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import { z } from 'zod';
import { MiddlewarePropsSchema } from '@project-lakechain/core/middleware';
import { TitanImageModel } from './image-model';
import {
  TextToImageTask,
  ImageInpaintingTask,
  ImageOutpaintingTask,
  ImageVariationTask,
  BackgroundRemovalTask,
  ColorGuidedGenerationTask
} from './tasks';

/**
 * Titan image generator properties schema.
 */
export const TitanImageGeneratorPropsSchema = MiddlewarePropsSchema.extend({

  /**
   * The image model to use.
   * @default TitanImageModel.TITAN_IMAGE_GENERATOR_V2
   */
  imageModel: z
    .instanceof(TitanImageModel)
    .default(TitanImageModel.TITAN_IMAGE_GENERATOR_V2),

  /**
   * The task to execute by the image model.
   */
  task: z.union([
    z.custom<TextToImageTask>(),
    z.custom<ImageInpaintingTask>(),
    z.custom<ImageOutpaintingTask>(),
    z.custom<ImageVariationTask>(),
    z.custom<BackgroundRemovalTask>(),
    z.custom<ColorGuidedGenerationTask>()
  ]),

  /**
   * The AWS region in which the model will
   * be invoked.
   */
  region: z
    .string()
    .optional()
});

// The type of the `TitanImageGeneratorProps` schema.
export type TitanImageGeneratorProps = z.infer<typeof TitanImageGeneratorPropsSchema>;
