/*
 * Copyright (C) 2023 Amazon.com, Inc. or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import path from 'path';
import serialize from 'serialize-javascript';

import * as cdk from 'aws-cdk-lib';
import * as lambda from 'aws-cdk-lib/aws-lambda';
import * as sources from 'aws-cdk-lib/aws-lambda-event-sources';
import * as iam from 'aws-cdk-lib/aws-iam';
import * as node from 'aws-cdk-lib/aws-lambda-nodejs';
import * as esbuild from 'esbuild';

import { Construct } from 'constructs';
import { ServiceDescription } from '@project-lakechain/core/service';
import { ComputeType } from '@project-lakechain/core/compute-type';
import { CacheStorage } from '@project-lakechain/core';
import { when } from '@project-lakechain/core/dsl';

import {
  TransformProps,
  TransformPropsSchema,
  TransformExpression
} from './definitions/opts';
import {
  Middleware,
  MiddlewareBuilder,
  LAMBDA_INSIGHTS_VERSION,
  NAMESPACE
} from '@project-lakechain/core/middleware';

/**
 * The service description.
 */
const description: ServiceDescription = {
  name: 'transform',
  description: 'A middleware allowing to transform documents on-the-fly.',
  version: '0.10.0',
  attrs: {}
};

/**
 * The maximum time the processing lambda
 * is allowed to run.
 */
const PROCESSING_TIMEOUT = cdk.Duration.minutes(1);

/**
 * The execution runtime for used compute.
 */
const EXECUTION_RUNTIME = lambda.Runtime.NODEJS_18_X;

/**
 * The default memory size to allocate for the compute.
 */
const DEFAULT_MEMORY_SIZE = 192;

/**
 * The name of the callable to invoke the
 * transform expression.
 */
const TRANSFORM_EXPRESSION_SYMBOL = '__callable';

/**
 * Builder for the `Transform` middleware.
 */
class TransformBuilder extends MiddlewareBuilder {
  private providerProps: Partial<TransformProps> = {};

  /**
   * Sets a function evaluated in the cloud to transform
   * input documents.
   * @param expression the transform expression.
   * @returns the builder instance.
   */
  public withTransformExpression(expression: TransformExpression | lambda.IFunction) {
    this.providerProps.expression = expression;
    return (this);
  }

  /**
   * @returns a new instance of the `Transform`
   * service constructed with the given parameters.
   */
  public build(): Transform {
    return (new Transform(
      this.scope,
      this.identifier, {
        ...this.providerProps as TransformProps,
        ...this.props
      }
    ));
  }
}

/**
 * A middleware acting allowing to transform input documents
 * using a transform expression.
 */
export class Transform extends Middleware {

  /**
   * The storage containing processed files.
   */
  public storage: CacheStorage;

  /**
   * The event processing lambda function.
   */
  public eventProcessor: lambda.IFunction;

  /**
   * The builder for the `Transform` service.
   */
  public static readonly Builder = TransformBuilder;

  /**
   * Provider constructor.
   */
  constructor(scope: Construct, id: string, props: TransformProps) {
    super(scope, id, description, {
      ...props,
      queueVisibilityTimeout: cdk.Duration.seconds(
        3 * PROCESSING_TIMEOUT.toSeconds()
      )
    });

    // Validating the properties.
    props = this.parse(TransformPropsSchema, props);

    ///////////////////////////////////////////
    ////////    Processing Storage      ///////
    ///////////////////////////////////////////

    this.storage = new CacheStorage(this, 'Storage', {
      encryptionKey: props.kmsKey
    });

    ///////////////////////////////////////////
    //////     Transform Expression      //////
    ///////////////////////////////////////////

    let expression: string;

    // If the transform expression is a Lambda function,
    // we give it the permission to read the processed
    // documents generated by the previous middlewares.
    if (props.expression instanceof lambda.Function) {
      expression = props.expression.functionArn;
      this.grantReadProcessedDocuments(props.expression);
    
    // If it is a function expression, we serialize it for the
    // processing function to evaluate it.
    } else if (typeof props.expression === 'function') {
      expression = this.serializeFn(props.expression);

    // Otherwise, we throw an error.
    } else {
      throw new Error(`
        Invalid or transform expression in transform middleware.
      `);
    }

    ///////////////////////////////////////////
    ///////    Processing Function      ///////
    ///////////////////////////////////////////

    this.eventProcessor = new node.NodejsFunction(this, 'Compute', {
      description: 'A function evaluating a transform expression.',
      entry: path.resolve(__dirname, 'lambdas', 'transform', 'index.js'),
      vpc: props.vpc,
      memorySize: props.maxMemorySize ?? DEFAULT_MEMORY_SIZE,
      timeout: PROCESSING_TIMEOUT,
      runtime: EXECUTION_RUNTIME,
      architecture: lambda.Architecture.ARM_64,
      tracing: lambda.Tracing.ACTIVE,
      environmentEncryption: props.kmsKey,
      logGroup: this.logGroup,
      insightsVersion: props.cloudWatchInsights ?
        LAMBDA_INSIGHTS_VERSION :
        undefined,
      environment: {
        POWERTOOLS_SERVICE_NAME: description.name,
        POWERTOOLS_METRICS_NAMESPACE: NAMESPACE,
        SNS_TARGET_TOPIC: this.eventBus.topicArn,
        PROCESSED_FILES_BUCKET: this.storage.id(),
        LAKECHAIN_CACHE_STORAGE: props.cacheStorage.id(),
        TRANSFORM_EXPRESSION_TYPE: props.expression instanceof lambda.Function ?
          'lambda' :
          'expression',
        TRANSFORM_EXPRESSION: expression,
        TRANSFORM_EXPRESSION_SYMBOL
      },
      bundling: {
        minify: true,
        externalModules: [
          '@aws-sdk/client-s3',
          '@aws-sdk/client-sns'
        ]
      }
    });

    // Allows this construct to act as a `IGrantable`
    // for other middlewares to grant the processing
    // lambda permissions to access their resources.
    this.grantPrincipal = this.eventProcessor.grantPrincipal;

    // If the expression is a lambda function, grant
    // the processing lambda the permission to invoke it.
    if (props.expression instanceof lambda.Function) {
      props.expression.grantInvoke(this.eventProcessor);
    }

    // Allow the function to publish to the SNS topic.
    this.eventBus.grantPublish(this.eventProcessor);

    // Allow the function to write to the S3 bucket.
    this.storage.grantReadWrite(this.eventProcessor);

    // Plug the SQS queue into the lambda function.
    this.eventProcessor.addEventSource(new sources.SqsEventSource(this.eventQueue, {
      batchSize: props.batchSize ?? 10,
      maxBatchingWindow: props.batchingWindow,
      reportBatchItemFailures: true
    }));

    super.bind();
  }

  /**
   * A helper used to serialize the different types of JavaScript
   * functions that the user can provide (e.g functions, arrow functions,
   * async functions, etc.) into a string.
   * This function also uses `esbuild` to validate the syntax of the
   * provided function and minify it.
   * @param fn the function to serialize.
   * @param opts the esbuild transform options.
   * @returns the serialized function.
   */
  private serializeFn(fn: TransformExpression, opts?: esbuild.TransformOptions): string {
    const res = esbuild.transformSync(`const ${TRANSFORM_EXPRESSION_SYMBOL} = ${serialize(fn)}\n`, {
      minify: true,
      ...opts
    });
    return (res.code);
  }

  /**
   * Allows a grantee to read from the processed documents
   * generated by this middleware.
   */
  grantReadProcessedDocuments(grantee: iam.IGrantable): iam.Grant {
    // Since this middleware simply passes through the data
    // from the previous middleware, we grant any subsequent
    // middlewares in the pipeline to have read access to the
    // data of all source middlewares.
    for (const source of this.sources) {
      source.grantReadProcessedDocuments(grantee);
    }
    // We also grant the grantee the permission to read from
    // the middleware storage containing transformed documents.
    this.storage.grantRead(grantee);
    return ({} as iam.Grant);
  }

  /**
   * @returns an array of mime-types supported as input
   * type by the data producer.
   */
  supportedInputTypes(): string[] {
    return ([
      '*/*'
    ]);
  }

  /**
   * @returns an array of mime-types supported as output
   * type by the data producer.
   */
  supportedOutputTypes(): string[] {
    return ([
      '*/*'
    ]);
  }

  /**
   * @returns the supported compute types by a given
   * middleware.
   */
  supportedComputeTypes(): ComputeType[] {
    return ([
      ComputeType.CPU
    ]);
  }

  /**
   * @returns the middleware conditional statement defining
   * in which conditions this middleware should be executed.
   * In this case, we want the middleware to only be invoked
   * when the document mime-type is supported, and the event
   * type is `document-created`.
   */
  conditional() {
    return (super
      .conditional()
      .and(when('type').equals('document-created'))
    );
  }
}

export { TransformExpression, Sdk, Environment } from './definitions/opts';
export { CloudEvent } from '@project-lakechain/sdk';
