//! Inherents for block-fees pallet
#![cfg_attr(not(feature = "std"), no_std)]

use codec::{Decode, Encode};
use domain_runtime_primitives::Balance;
use sp_inherents::{InherentIdentifier, IsFatalError};

/// Block-fees inherent identifier.
pub const INHERENT_IDENTIFIER: InherentIdentifier = *b"blockfee";

#[derive(Debug, Encode, Decode)]
pub enum InherentError {
    IncorrectConsensusChainByteFee,
}

impl IsFatalError for InherentError {
    fn is_fatal_error(&self) -> bool {
        true
    }
}

/// The type of the inherent.
pub type InherentType = Balance;

/// Provides the set code inherent data.
#[cfg(feature = "std")]
pub struct InherentDataProvider {
    data: InherentType,
}

#[cfg(feature = "std")]
impl InherentDataProvider {
    /// Create new inherent data provider from the given `data`.
    pub fn new(data: InherentType) -> Self {
        Self { data }
    }

    /// Returns the `data` of this inherent data provider.
    pub fn data(&self) -> &InherentType {
        &self.data
    }
}

#[cfg(feature = "std")]
#[async_trait::async_trait]
impl sp_inherents::InherentDataProvider for InherentDataProvider {
    async fn provide_inherent_data(
        &self,
        inherent_data: &mut sp_inherents::InherentData,
    ) -> Result<(), sp_inherents::Error> {
        inherent_data.put_data(INHERENT_IDENTIFIER, &self.data)
    }

    async fn try_handle_error(
        &self,
        identifier: &InherentIdentifier,
        error: &[u8],
    ) -> Option<Result<(), sp_inherents::Error>> {
        if *identifier != INHERENT_IDENTIFIER {
            return None;
        }

        let error = InherentError::decode(&mut &*error).ok()?;

        Some(Err(sp_inherents::Error::Application(Box::from(format!(
            "{error:?}"
        )))))
    }
}
