/*
 *  arg.h
 *  artoolkitX
 *
 *	Graphics Subroutines for artoolkitX.
 *
 *  This file is part of artoolkitX.
 *
 *  artoolkitX is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  artoolkitX is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with artoolkitX.  If not, see <http://www.gnu.org/licenses/>.
 *
 *  As a special exception, the copyright holders of this library give you
 *  permission to link this library with independent modules to produce an
 *  executable, regardless of the license terms of these independent modules, and to
 *  copy and distribute the resulting executable under terms of your choice,
 *  provided that you also meet, for each linked independent module, the terms and
 *  conditions of the license of that module. An independent module is a module
 *  which is neither derived from nor based on this library. If you modify this
 *  library, you may extend this exception to your version of the library, but you
 *  are not obligated to do so. If you do not wish to do so, delete this exception
 *  statement from your version.
 *
 *  Copyright 2018 Realmax, Inc.
 *  Copyright 2015 Daqri, LLC.
 *  Copyright 2011-2015 ARToolworks, Inc.
 *
 *  Author(s): Philip Lamb
 *
 */

/*!
    @file arg.h
    @brief A collection of useful graphics routines for artoolkitX.
    @details
        arg is the preferred means for drawing camera video images acquired from
        artoolkitX's video libraries. It includes optimized texture handling, and
        a variety of flexible drawing options.
        
        Implementations are provided for OpenGL 1.5 and later, OpenGL 3.1 and
        later, and OpenGL ES 2.0 and later.
        
        The chosen API is determined by the system and context available at the
        time the first call is made to arglSetupForCurrentContext()/
 
        arg does not depend on any particular window or event handling system.
  
    @copyright 2011-2015 Daqri, LLC.
    @version 5.4.0
 */

#ifndef __arg_h__
#define __arg_h__

// ============================================================================
//  Public includes.
// ============================================================================

#include <ARX/AR/ar.h>      // ARUint8, AR_PIXEL_FORMAT, ARParam, arParamObserv2Ideal()

#ifdef __cplusplus
extern "C" {
#endif
    
// ============================================================================
//  Public types and definitions.
// ============================================================================

#ifdef _WIN32
#  ifdef ARG_STATIC
#    define ARG_EXTERN
#  else
#    ifdef ARX_EXPORTS
#      define ARG_EXTERN __declspec(dllexport)
#    else
#      define ARG_EXTERN __declspec(dllimport)
#    endif
#  endif
#else
#  define ARG_EXTERN
#endif

/*!
    @typedef ARGL_CONTEXT_SETTINGS_REF
    @brief Opaque type to hold ARGL settings for a given OpenGL context.
    @details
        An OpenGL context is an implementation-defined structure which
        keeps track of OpenGL state, including textures and display lists.
        Typically, individual OpenGL windows will have distinct OpenGL
        contexts assigned to them by the host operating system.
 
        As arg uses textures, buffers, and programs, it must be able to
        track which OpenGL context a given texture, buffer, or program it is using
        belongs to. This is especially important when arg is being used to
        draw into more than one window (and therefore more than one context.)
 
        Basically, functions which depend on OpenGL state, will require an
        ARGL_CONTEXT_SETTINGS_REF to be passed to them. An ARGL_CONTEXT_SETTINGS_REF
        is generated by setting the current OpenGL context (e.g. if using GLUT,
        you might call glutSetWindow()) and then calling arglSetupForCurrentContext().
        When you have finished using ARGL in a given context, you should call
        arglCleanup(), passing in an ARGL_CONTEXT_SETTINGS_REF, to free the
        memory used by the settings structure.
    @since First appeared in ARToolKit 2.68.
 */
typedef struct _ARGL_CONTEXT_SETTINGS *ARGL_CONTEXT_SETTINGS_REF;

/*!
    @brief Underlying rendering API being used in implementation.
 */
typedef enum {
    ARG_API_None,       ///< No API currently selected.
    ARG_API_GL,         ///< OpenGL 1.5 or later.
    ARG_API_GL3,        ///< OpenGL 3.1 or later.
    ARG_API_GLES2       ///< OpenGL ES 2.0 or later.
} ARG_API;

// ============================================================================
//  Public functions.
// ============================================================================

/*!
    @brief Initialise the arg library for the current OpenGL context.
    @details
        This function performs required setup of the arg library
        for the current OpenGL context and must be called before any other argl*()
        functions are called for this context.
 
        An OpenGL context holds all of the state of the OpenGL machine, including
        textures and display lists etc. There will usually be one OpenGL context
        for each window displaying OpenGL content.
 
        Other argl*() functions whose operation depends on OpenGL state will
        require an ARGL_CONTEXT_SETTINGS_REF. This is just so that
        they can keep track of per-context variables.
 
        You should call arglCleanup() passing in the ARGL_CONTEXT_SETTINGS_REF
        when you have finished with the library for this context.
    @param cparam Pointer to a set of ARToolKit camera parameters for the
        current video source. The size of the source image is taken from the
        fields xsize and ysize of the ARParam structure pointed to. Also, the
        field dist_factor of the ARParam structure pointed to will be taken as
        the amount to un-warp the supplied image.
 
        Prior to ARToolKit v5.1.5, the ARParam structure pointed to had to remain
        valid in memory for the duration of the ARGL_CONTEXT_SETTINGS_REF's usage.
        As of ARToolKit v5.1.5, the ARParam structure pointed to is copied, and
        no longer need be maintained.
    @param pixelFormat The type and format of pixels of the images to be drawn by the
        arg library. This value can be changed later by using arglPixelFormatSet()
    @result An ARGL_CONTEXT_SETTINGS_REF. See the documentation for this type for more info.
    @since First appeared in ARToolKit 2.68.
*/
ARG_EXTERN ARGL_CONTEXT_SETTINGS_REF arglSetupForCurrentContext(const ARParam *cparam, AR_PIXEL_FORMAT pixelFormat);

/*!
    @brief Provide an ARHandle structure from which to extract information about debug mode.
    @details
        This provides support for the debug mode in libAR. If arg has been compiled
        with debug mode support, the passed in ARHandle is used as the source
        of whether debug mode is enabled or disabled. If enabled, the binarized
        debug image will be drawn instead of the input image.
    @param arHandle A pointer to an ARHandle structure from which to extract debug info,
        or NULL if debug functionality is not required. Default is NULL.
    @result TRUE if the mode was set, FALSE if an error occurred.
    @since First appeared in ARToolKit 4.5.12.
    @see arglPixelBufferDataUpload
*/
ARG_EXTERN int arglSetupDebugMode(ARGL_CONTEXT_SETTINGS_REF contextSettings, ARHandle *arHandle);


/*!
    @brief Free memory used by arg associated with the specified context.
    @details
        Should be called after no more argl* functions are needed, in order
        to prevent memory leaks etc.
 
        The library can be setup again for the context at a later time by calling
        arglSetupForCurrentContext() again.
    @param contextSettings A reference to ARGL's settings for an OpenGL
        context, as returned by arglSetupForCurrentContext(). If NULL, arglCleanup
        performs no operation.
    @since First appeared in ARToolKit 2.68.
*/
ARG_EXTERN void arglCleanup(ARGL_CONTEXT_SETTINGS_REF contextSettings);

/*!
    @brief Display an ARVideo image, by drawing it using OpenGL.
    @details
        This function draws an image from an ARVideo source to the current
        OpenGL context. This operation is most useful in video see-through
        augmented reality applications for drawing the camera view as a
        background image, but can also be used in other ways.
 
        An undistorted image is drawn with the lower-left corner of the
        bottom-left-most pixel at OpenGL screen coordinates (0,0), and the
        upper-right corner of the top-right-most pixel at OpenGL screen
        coodinates (x * zoom, y * zoom), where x and y are the values of the
        fields cparam->xsize and cparam->ysize (see below) and zoom is the
        value of the parameter zoom (also see below). If cparam->dist_factor
        indicates that an un-warping correction should be applied, the actual
        coordinates will differ from the values specified here. 
 
        OpenGL state: Drawing is performed with depth testing and lighting
        disabled, and thus leaves the the depth buffer (if any) unmodified. If
        pixel transfer is by texturing and the drawing is done in replacement
        texture environment mode. The depth test enable and lighting enable
        state and the texture environment mode are NOT restored before the
        function returns.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context. It
        is the callers responsibility to make sure that the current context at the
        time arglDisplayImage() is called matches that under which contextSettings
        was created.
    @param viewport OpenGL-style viewport to be applied prior to drawing, or NULL
        in which case the viewport will not be set prior to drawing.
    @since First appeared in ARToolKit 2.68.
*/
ARG_EXTERN void arglDispImage(ARGL_CONTEXT_SETTINGS_REF contextSettings, int32_t viewport[4]);

/*!
    @brief Set compensation for camera lens distortion in arglDispImage to off or on.
    @details
        By default, arglDispImage compensates for the distortion of the camera's
        acquired image caused by the lens when it draws. By calling this function
        with enabled = FALSE, this compensation will be disabled in the specified
        drawing context. It may be re-enabled at any time.
        This function is useful if you need to draw an image, but do not know the
        extent of the camera's lens distortion (such as during distortion calibration).
        While distortion compensation is disabled, the dist_factor[] array in a
        the camera cparam structure passed to arglDispImage is ignored.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context. 
    @param enable TRUE to enabled distortion compensation, FALSE to disable it.
        The default state for new contexts is enable = TRUE.
    @result TRUE if the distortion value was set, FALSE if an error occurred.
    @since First appeared in ARToolKit 2.71.
*/
ARG_EXTERN int arglDistortionCompensationSet(ARGL_CONTEXT_SETTINGS_REF contextSettings, int enable);

/*!
    @brief Enquire as to the enable state of camera lens distortion compensation in arglDispImage.
    @details
        By default, arglDispImage compensates for the distortion of the camera's
        acquired image caused by the lens when it draws. This function enquires
        as to whether arglDispImage is currently doing compensation or not.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context. 
    @param enable Pointer to an integer value which will be set to TRUE if distortion
        compensation is enabled in the specified context, or FALSE if it is disabled.
    @result TRUE if the distortion value was retreived, FALSE if an error occurred.
    @since First appeared in ARToolKit 2.71.
 */
ARG_EXTERN int arglDistortionCompensationGet(ARGL_CONTEXT_SETTINGS_REF contextSettings, int *enable);

/*!
    @brief Set the current video image drawing scalefactor.
    @param zoom The amount to scale the video image up or down. To draw the video
        image double size, use a zoom value of 2.0. To draw the video image
        half size use a zoom value of 0.5.
 */
ARG_EXTERN int arglSetPixelZoom(ARGL_CONTEXT_SETTINGS_REF contextSettings, float zoom);

/*!
    @brief Retrieve the current video image drawing scalefactor.
    @param zoom The amount to scale the video image up or down. To draw the video
        image double size, use a zoom value of 2.0. To draw the video image
        half size use a zoom value of 0.5.
 */
ARG_EXTERN int arglGetPixelZoom(ARGL_CONTEXT_SETTINGS_REF contextSettings, float *zoom);
    
/*!
    @brief Set the format of pixel data which will be passed to arglDispImage*()
    @details (description)
        In arg, the format of the pixels (i.e. the arrangement of components
        within each pixel) can be changed at runtime. Use this function to inform
        arg the format the pixels being passed to arglDispImage*() functions
        are in. This setting applies only to the context passed in parameter
        contextSettings. The default format is determined by the value passed to
        arglSetupForCurrentContext.
        Usually, image data is passed in directly from images generated by ARVideo,
        and so you should ensure that ARVideo is generating pixels of the same format.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context. 
    @param format A symbolic constant for the pixel format being set. See
        AR_PIXEL_FORMAT in ar.h for a list of all possible formats.
    @result TRUE if the pixel format value was set, FALSE if an error occurred.
    @since First appeared in ARToolKit 2.71.
*/
ARG_EXTERN int arglPixelFormatSet(ARGL_CONTEXT_SETTINGS_REF contextSettings, AR_PIXEL_FORMAT format);

/*!
    @brief Get the format of pixel data in which arglDispImage*() is expecting data to be passed.
    @details This function enquires as to the current format of pixel data being
        expected by the arglDispImage*() functions. The default format is determined by
        the value passed to arglSetupForCurrentContext.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context. 
    @param format A symbolic constant for the pixel format in use. See
        AR_PIXEL_FORMAT in ar.h for a list of all possible formats.
    @param size The number of bytes of memory occupied per pixel, for the given format.
    @result TRUE if the pixel format and size values were retreived, FALSE if an error occurred.
    @since First appeared in ARToolKit 2.71.
*/
ARG_EXTERN int arglPixelFormatGet(ARGL_CONTEXT_SETTINGS_REF contextSettings, AR_PIXEL_FORMAT *format, int *size);

/*!
    @brief   Find out whether ARGL is rotating all OpenGL drawing by 90 degrees.
    @details
        On some OpenGL ES devices, it may be desirable to rotate all OpenGL drawing by 90
        degrees in the window coordinate system, in order to swap the horizontal and
        vertical axes of the device. This may be a higher performance solution than
        manually swapping rows and columns of submitted data.
 
        This function queries the current state of whether such a rotation is being performed by ARGL or not.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context.
    @result    TRUE if a 90 degree rotation is enabled, FALSE if it is disabled.
*/
ARG_EXTERN int arglGetRotate90(ARGL_CONTEXT_SETTINGS_REF contextSettings);

/*!
    @brief   Set or unset a 90 degree rotation in all OpenGL drawing performed by ARGL.
    @details
        On some OpenGL ES devices, it may be desirable to rotate all OpenGL drawing by 90
        degrees in the window coordinate system, in order to swap the horizontal and
        vertical axes of the device. This may be a higher performance solution than
        manually swapping rows and columns of submitted data.
 
        This function enables or disables such an axis swap in ARGL with very little 
        performance cost, by introducing a 90-degree rotation into the OpenGL projection matrix.
 
        By default, 90 degree rotation is DISABLED.
    @param contextSettings A reference to ARGL's settings for the current OpenGL.
        context, as returned by arglSetupForCurrentContext() for this context.
    @param rotate90 Set to TRUE or 1 to enable 90 degree rotation, FALSE to disable.
*/
ARG_EXTERN void arglSetRotate90(ARGL_CONTEXT_SETTINGS_REF contextSettings, int rotate90);

ARG_EXTERN int arglGetFlipH(ARGL_CONTEXT_SETTINGS_REF contextSettings);
ARG_EXTERN void arglSetFlipH(ARGL_CONTEXT_SETTINGS_REF contextSettings, int flipH);
ARG_EXTERN int arglGetFlipV(ARGL_CONTEXT_SETTINGS_REF contextSettings);
ARG_EXTERN void arglSetFlipV(ARGL_CONTEXT_SETTINGS_REF contextSettings, int flipV);

/*!
    @brief   Specify a desired pixel buffer size larger than the camera image size.
    @details
        By default, the pixel buffer accepted by function arglPixelBufferDataUpload()
        is assumed to be tightly packed, row-major array of dimensions
        equal to the calibrated camera image size (as passed in the fields arParam.xsize and arParam.ysize
        of the ARHandle submitted to arglSetupForCurrentContext().

        The pixel data buffer submitted may, under some circumstances be allowed to be larger than the
        values of the calibrated camera image size (i.e. padded). This may only occur when the
        underlying OpenGL ES impementation does not support non-power-of-two textures. For the Apple
        iPhone family, this applies to the iPhone, the iPhone 3G, the iPod Touch 1st and 2nd Generation.
        The iPhone 3GS and the iPod Touch 3rd Generation support non-power-of-two textures, and thus
        padding of the pixel data buffer is not supported on these devices. If padding is desired,
        the desired buffer size must be submitted using this function. Check that the
        result from this function is TRUE before attempting to upload such a padded buffer.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context.
    @param bufWidth The desired buffer width, in pixels.
    @param bufHeight The desired buffer height, in pixels.
    @result TRUE if the desired buffer size is supported, FALSE otherwise.
*/
ARG_EXTERN int arglPixelBufferSizeSet(ARGL_CONTEXT_SETTINGS_REF contextSettings, int bufWidth, int bufHeight);

/*!
    @brief   Query the size of pixel data buffers expected by arglPixelBufferDataUpload().
    @details
        See function arglPixelBufferSizeSet() for a full discussion.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context.
    @param bufWidth A pointer to int, which will be filled out with the buffer width, in pixels, or NULL if this value is not required.
    @param bufHeight A pointer to int, which will be filled out with the buffer height, in pixels, or NULL if this value is not required..
    @result TRUE if the buffer size was successfully queried, FALSE otherwise.
 */
ARG_EXTERN int arglPixelBufferSizeGet(ARGL_CONTEXT_SETTINGS_REF contextSettings, int *bufWidth, int *bufHeight);

/*!
    @brief   Upload a buffer of pixel data to an OpenGL texture for later use by arglDispImage().
    @details
        ARGL provides the ability to perform distortion-compensated texturing of a camera video image
        into the OpenGL context, for the purpose of video-see through augmented reality. This function
        uploads the camera image data to an OpenGL texture, ready for later drawing using the function
        arglDispImage().
 
        If a pointer to an ARParam structure has been passed in to function arglSetupDebugMode, then
        a check will be made whether debug mode is enabled, and if it is, then the buffer pointed to
        by this function will be ignored, and instead the debug image will be loaded for texturing.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context.
    @param bufDataPtr0 A pointer to the pixel buffer, which is a block of memory from which texture
        data will be read.
        
        The layout of pixel data in the memory pointed to by bufDataPtr is assumed to be
        specified by the value of pixelFormat in the ARHandle submitted to arglSetupForCurrentContext(),
        but can be changed by calling arglPixelFormatSet() and/or arglPixelBufferSizeSet().
 
        By default, the pixel buffer is assumed to be tightly packed, row-major array of dimensions
        equal to the calibrated camera image size (as passed in the fields arParam.xsize and arParam.ysize
        of the ARHandle submitted to arglSetupForCurrentContext().
 
        The pixel data buffer submitted may, under some circumstances be allowed to be larger than the
        values of the calibrated camera image size (i.e. padded). See the discussion section of the
        documentation for function arglPixelBufferSizeSet() for more information.
 
        This may only occur when the
        underlying OpenGL ES impementation does not support non-power-of-two textures. For the Apple
        iPhone family, this applies to the iPhone, the iPhone 3G, the iPod Touch 1st and 2nd Generation.
        The iPhone 3GS and the iPod Touch 3rd Generation support non-power-of-two textures, and thus
        padding of the pixel data buffer is not supported on these devices. If padding is desired,
        the desired buffer size must be submitted using arglPixelBufferSizeSet(). Check that the
        result from this function is TRUE before attempting to upload such a padded buffer.
    @param bufDataPtr1 NULL for single-plane (interleaved) textures, otherwise for bi-planar textures,
        a pointer to the pixel buffer, which is a block of memory from which texture
        data will be read for the second plane containing the CbCr components. This plane must have
        dimensions of half the size of plane 0, and 2-bytes per pixel (1 byte Cb, 1 byte Cr).
    @result TRUE if the pixel buffer was successfully uploaded to OpenGL, FALSE otherwise.
    @see arglSetupDebugMode
*/
ARG_EXTERN int arglPixelBufferDataUploadBiPlanar(ARGL_CONTEXT_SETTINGS_REF contextSettings, ARUint8 *bufDataPtr0, ARUint8 *bufDataPtr1);
#define arglPixelBufferDataUpload(contextSettings,bufDataPtr) arglPixelBufferDataUploadBiPlanar(contextSettings,bufDataPtr,NULL)
    
/*!
    @brief   Checks for the presence of an OpenGL capability by version or extension.
    @details
        Checks for the presence of an OpenGL capability by version or extension.
        The test returns true if EITHER the OpenGL driver's OpenGL implementation
        version meets or exceeds a minimum value (passed in in minVersion) OR
        if an OpenGL extension identifier passed in as a character string
        is non-NULL, and is found in the current driver's list of supported extensions.
    @param contextSettings A reference to ARGL's settings for the current OpenGL
        context, as returned by arglSetupForCurrentContext() for this context.
    @param      minVersion
        A binary-coded decimal (i.e. version 1.0 is represented as 0x0100) version number.
        If minVersion is zero, (i.e. there is no version of OpenGL with this extension in core)
        the version test will always fail, and the result will only be true if the extension
        string test passes.
    @param      extension A string with an extension name to search the drivers extensions
        string for. E.g. "GL_EXT_texture". If NULL, the extension name test will always fail,
        and the result will only be true if the version number test passes.
    @result     TRUE If either of the tests passes, or FALSE if both fail.
    @since First available in ARToolKit 4.5.0.
 */
ARG_EXTERN int arglGLCapabilityCheck(ARGL_CONTEXT_SETTINGS_REF contextSettings, const uint16_t minVersion, const unsigned char *extension);

#ifdef __cplusplus
}
#endif

#endif // !__arg_h__
